package org.clazzes.login.oauth;

import java.net.URI;
import java.net.URISyntaxException;
import java.time.Clock;
import java.time.Instant;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.clazzes.login.jbo.jwt.JWKPubKey;
import org.clazzes.login.oauth.impl.RobotCredentialsSupplier;
import org.clazzes.util.sec.OAuthCredentials;

public class DomainManager {
    @SuppressWarnings("unused")
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(DomainManager.class);

    private final DomainConfig domainConfig;
    private final Clock clock;
    private final OAuthHttpClient oAuthHttpClient;
    private final TokenValidator tokenValidator;
    private RobotCredentialsSupplier credentialsSupplier;

    private static final class OpenIdConfigurationData {
        private final Map<String, Object> configuration;
        private final Map<String, JWKPubKey> openIdKeys;
        private final Instant expiryTime;
        // The configurationLocation at the time that this data was fetched.
        private final URI configurationLocation;
        public OpenIdConfigurationData(Map<String, Object> configuration, Map<String, JWKPubKey> openIdKeys,
                Instant expiryTime, URI configurationLocation) {
            this.configuration = configuration;
            this.openIdKeys = openIdKeys;
            this.expiryTime = expiryTime;
            this.configurationLocation = configurationLocation;
        }
        public Map<String, Object> getConfiguration() {
            return this.configuration;
        }
        public Map<String, JWKPubKey> getOpenIdKeys() {
            return this.openIdKeys;
        }
        public Instant getExpiryTime() {
            return this.expiryTime;
        }
        public URI getConfigurationLocation() {
            return this.configurationLocation;
        }
        public OpenIdConfigurationData withExpiry(Instant expiryTimeOverride) {
            return new OpenIdConfigurationData(this.configuration,this.openIdKeys, expiryTimeOverride,this.configurationLocation);
        }
    }

    private OpenIdConfigurationData openIdConfiguration;

    public DomainManager(DomainConfig domainConfig, Clock clock, OAuthHttpClient oAuthHttpClient,
            TokenValidator tokenValidator) {
        this.domainConfig = domainConfig;
        this.clock = clock;
        this.oAuthHttpClient = oAuthHttpClient;
        this.tokenValidator = tokenValidator;
    }

    public DomainConfig getDomainConfig() {
        return this.domainConfig;
    }

    private OpenIdConfigurationData fetchNewOpenIdConfigurationData(ZonedDateTime now, URI configurationLocation, OpenIdConfigurationData old) {
        Map<String, Object> configuration;
        Map<String, JWKPubKey> openIdKeys;
        try {
            configuration = this.oAuthHttpClient.loadConfiguration(configurationLocation);

            URI jwksUri = URI.create(configuration.get("jwks_uri").toString());

            openIdKeys = this.oAuthHttpClient.loadPublicKeys(jwksUri)
                .stream()
                .collect(Collectors.toMap(key -> key.getPubKeyInfo().getKeyId(), Function.identity()));
        } catch (Exception e) {
            if (old == null) {
                throw new RuntimeException("Failed to fetch openid configuration.", e);
            } else {
                log.warn("Failed to refresh openid configuration", e);
                return old.withExpiry(now.plusMinutes(1).toInstant());
            }
        }

        Instant expiry = now
            .withHour(0)
            .withMinute(0)
            .withNano(0)
            .plusDays(1)
            // calculate between 0:00 and 2:00 on a random basis.
            .plus((long)(Math.random() * 7200000L), ChronoUnit.MILLIS)
            .toInstant();

        return new OpenIdConfigurationData(configuration, openIdKeys, expiry, configurationLocation);

    }

    private synchronized Optional<OpenIdConfigurationData> getOpenIdCache() {
        if (this.domainConfig.getConfigurationLocation() == null) {
            return Optional.empty();
        }

        ZonedDateTime now = ZonedDateTime.now(this.clock);

        if (this.openIdConfiguration != null
            && this.openIdConfiguration.getConfigurationLocation()
                .equals(this.domainConfig.getConfigurationLocation())
            && this.openIdConfiguration.getExpiryTime().isAfter(now.toInstant())) {
            return Optional.of(this.openIdConfiguration);
        }

        return Optional.of(this.openIdConfiguration = this.fetchNewOpenIdConfigurationData(now, this.domainConfig.getConfigurationLocation(),this.openIdConfiguration));
    }

    public Map<String, Object> getOpenIdConfiguration() {
        return this.getOpenIdCache()
            .map(OpenIdConfigurationData::getConfiguration)
            .orElse(null);
    }

    public URI getOpenIdLocation(String key) throws URISyntaxException, IllegalStateException {
        Map<String, Object> openIdCfg = this.getOpenIdConfiguration();

        if (openIdCfg == null) {
            throw new IllegalStateException();
        }

        Object uri_o = openIdCfg.get(key);

        if (uri_o == null) {
            return null;
        }

        return new URI(uri_o.toString());
    }

    public Map<String, JWKPubKey> getOpenIdKeys() {
        return this.getOpenIdCache()
            .map(OpenIdConfigurationData::getOpenIdKeys)
            .orElse(null);
    }

    public URI getTokenUri() throws IllegalStateException, URISyntaxException {
        URI tokenUri = this.getDomainConfig().getTokenLocation();

        if (tokenUri == null) {

            tokenUri = this.getOpenIdLocation("token_endpoint");
        }

        if (tokenUri == null) {
            throw new IllegalStateException("Unable to get token endpoint for domain ["+this.getDomainConfig().getDomain()+"]");
        }

        return tokenUri;
    }

    public String getDomain() {
        return this.getDomainConfig().getDomain();
    }

    public synchronized Supplier<? extends OAuthCredentials> getCredentialsSupplier() {
        if (!this.getDomainConfig().isRobot()) {
            throw new IllegalStateException("Tries to get robot credentials supplier for domain that doesn't support robot tokens");
        }

        if (this.credentialsSupplier == null) {
            this.credentialsSupplier =
                new RobotCredentialsSupplier(this.oAuthHttpClient,
                                             this.tokenValidator,
                                             this,this.clock);
        }

        return this.credentialsSupplier;
    }
}
