/***********************************************************
 * $Id$
 *
 * http://www.clazzes.org
 *
 * Created: 02.04.2011
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.net.PasswordAuthentication;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Dictionary;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.clazzes.util.lang.Triple;
import org.clazzes.util.sec.OAuthCredentials;
import org.osgi.framework.Bundle;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.blueprint.container.ServiceUnavailableException;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.cm.ManagedService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationService implements ManagedService {

    private static final Logger log = LoggerFactory.getLogger(ConfigurationService.class);

    public static final String CONFIG_PID = "org.clazzes.login.oauth";

    private OAuthHttpClient oauthHttpClient;
    private TokenValidator tokenValidator;
    private BiFunction<String,String,String> secretsService;
    private Bundle blueprintBundle;

    class DomainConfigRecord {
        private DomainManager domainManager;
        private ServiceRegistration<Supplier<? extends OAuthCredentials>> robotSupplierRegistration;
        public DomainConfigRecord(DomainManager domainManager,
                ServiceRegistration<Supplier<? extends OAuthCredentials>> robotSupplierRegistration) {
            this.domainManager = domainManager;
            this.robotSupplierRegistration = robotSupplierRegistration;
        }
        public DomainManager getDomainManager() {
            return this.domainManager;
        }
        public void setDomainManager(DomainManager domainManager) {
            this.domainManager = domainManager;
        }
        public ServiceRegistration<Supplier<? extends OAuthCredentials>> getRobotSupplierRegistration() {
            return this.robotSupplierRegistration;
        }
        public void setRobotSupplierRegistration(
                ServiceRegistration<Supplier<? extends OAuthCredentials>> robotSupplierRegistration) {
            this.robotSupplierRegistration = robotSupplierRegistration;
        }
    }

    /**
     * A mapping from domain names to domain configuration.
     */
    private final Map<String,DomainConfigRecord> domainConfigurations;
    private final List<String> guiDomains; // list of domains with user flow.

    public ConfigurationService() {
        this.domainConfigurations = new HashMap<>();
        this.guiDomains = new ArrayList<String>();
    }


    private void stopOpenIdJobs() {

        for (Map.Entry<String,DomainConfigRecord> e : this.domainConfigurations.entrySet()) {

            ServiceRegistration<Supplier<? extends OAuthCredentials>> robotReg = e.getValue().getRobotSupplierRegistration();

            if (robotReg != null) {
                if (log.isDebugEnabled()) {
                    log.debug("Unregistering robot token supplier for domain [{}]",e.getKey());
                }
                robotReg.unregister();
                e.getValue().setRobotSupplierRegistration(null);
            }
        }
    }

    @SuppressWarnings("unchecked")
    private void startOpenIdJobs() {
        for (Map.Entry<String,DomainConfigRecord> e : this.domainConfigurations.entrySet()) {
            DomainManager domainManager = e.getValue().getDomainManager();

            if (domainManager.getDomainConfig().isRobot()) {

                Supplier<? extends OAuthCredentials> supplier = domainManager.getCredentialsSupplier();

                Dictionary<String,String> props = new Hashtable<String,String>();

                props.put("login.mechanism","org.clazzes.login.oauth");
                props.put("login.domain",e.getKey());
                props.put("login.credentials","org.clazzes.util.sec.OAuthCredentials");

                if (log.isDebugEnabled()) {
                    log.debug("Registering robot token supplier for domain [{}] with properties [{}]",
                              e.getKey(),props);
                }

                e.getValue().setRobotSupplierRegistration(
                    (ServiceRegistration<Supplier<? extends OAuthCredentials>>)
                    this.blueprintBundle.getBundleContext().registerService(
                        Supplier.class.getName(),
                        supplier,
                        props)
                );
            }
        }
    }

    private <K,V> Map<K, V> dictToMap(Dictionary<K, V> dict) {
        Map<K, V> ret = new HashMap<K, V>();

        Enumeration<K> keys = dict.keys();
        while (keys.hasMoreElements()) {
            K key = keys.nextElement();
            ret.put(key, dict.get(key));
        }

        return ret;
    }

    private String getSecret(Map<String, String> properties, String key) {
        String unresolvedValue = properties.get(key);
        if (unresolvedValue == null) {
            return null;
        }

        if (unresolvedValue.startsWith("secret::")) {
            String secretKey = unresolvedValue.substring(8);

            try {
                String value = this.secretsService.apply(CONFIG_PID,secretKey);
                log.info("Resolved {} secret [{}] from OSGi secrets service.",key,secretKey);
                return value;
            }
            catch (ServiceUnavailableException e) {
                log.warn("Cannot resolve {} secret with no secrets service available.", key);
                return null;
            }
        } else {
            return unresolvedValue;
        }
    }

    private URI getUri(Map<String, String> properties, String key) throws ConfigurationException {
        String stringValue = properties.get(key);
        if (stringValue == null) {
            return null;
        }

        try {
            return new URI(stringValue);
        } catch (URISyntaxException e) {
            throw new ConfigurationException(key,
            "Invalid format of "+key+" URI ["+
                stringValue +"]: Invalid URI syntax: "+e.getMessage());

        }
    }

	private URI getFaviconLocation(Map<String, String> domainProperties, URI configurationLocation,
			URI authorizationLocation) throws ConfigurationException {

		URI faviconLocation = getUri(domainProperties, "faviconLocation");

        if (faviconLocation == null) {
            try {
                if (configurationLocation != null) {
                    faviconLocation = new URI(configurationLocation.getScheme(),null,
                    configurationLocation.getHost(),configurationLocation.getPort(),"/favicon.ico",null,null);
                }
                else if (authorizationLocation != null) {
                    faviconLocation = new URI(authorizationLocation.getScheme(),null,
                    authorizationLocation.getHost(),authorizationLocation.getPort(),"/favicon.ico",null,null);
                }

            } catch (URISyntaxException e) {
                throw new ConfigurationException("faviconLocation", "Couldn't guess the faviconLocation from the configurationLocation or the authoizationLocation, please specify it explicitly.");
            }
        }

		return faviconLocation;
	}

    private static PasswordAuthentication getPWAuth(String lbl, String user, String password) {

        if (user != null && password == null) {
            log.warn("No password for OAuth user role [{}] given, authentication will not work.",lbl);
        }

        if (user == null || password ==  null) {
            return null;
        }

        return new PasswordAuthentication(user,password.toCharArray());
    }

    private DomainConfig extractConfig(String domain, Map<String, String> domainProperties) throws ConfigurationException {
        String label = domainProperties.get("label");
        String clientId = domainProperties.get("clientId");
        String clientpw = getSecret(domainProperties, "clientPassword");

        PasswordAuthentication clientAuth = getPWAuth("client",clientId,clientpw);

        RobotGrantType robotGrantType = Optional.ofNullable(domainProperties.get("robotGrantType"))
            .map(RobotGrantType::valueOf)
            .orElse(null);

        String robotClientId = domainProperties.get("robotClientId");
        String robotClientPassword = getSecret(domainProperties, "robotClientPassword");

        PasswordAuthentication robotAuth = robotGrantType != null && robotClientId != null
            ? getPWAuth("robotClient",robotClientId,robotClientPassword)
            : null;

        String robotUserId = domainProperties.get("robotUserId");
        String robotUserPassword = getSecret(domainProperties, "robotUserPassword");

        PasswordAuthentication robotUserCredentials = robotGrantType == RobotGrantType.password && robotUserId != null
            ? getPWAuth("robotUser",robotUserId,robotUserPassword)
            : null;

        URI configurationLocation = getUri(domainProperties, "configurationLocation");
        URI authorizationLocation = getUri(domainProperties, "authorizationLocation");
        URI tokenLocation = getUri(domainProperties, "tokenLocation");
        URI userLocation = getUri(domainProperties, "userLocation");

        URI faviconLocation = getFaviconLocation(domainProperties, configurationLocation, authorizationLocation);

        String scope = domainProperties.get("scope");
        String prompt = domainProperties.get("prompt");
        String accessType = domainProperties.get("accessType");
        String resource = domainProperties.get("resource");
        String groupRoleResource = domainProperties.get("groupRoleResource");
        String optionsString = domainProperties.get("options");
        EnumSet<ConfigOptions> options;
        try {
            options = ConfigOptions.parseOptions(optionsString);
        } catch (IllegalArgumentException e) {
            throw new ConfigurationException("options",
            "Invalid format of options ["+
                optionsString +"]: Must be a comma separated list out of "+EnumSet.allOf(ConfigOptions.class));
        }
        Map<String, String> appUsers = JsonConfigParser.parseAppUsers(domainProperties.get("appUsers"));
        String robotScope = domainProperties.get("robotScope");


        if (robotAuth == null && clientAuth == null) {
            throw new ConfigurationException("clientId", "Neither a user nor a robot ClientID has been specified.");
        }

        if (authorizationLocation == null && configurationLocation == null && clientAuth != null) {
            // we need an authorization location for user flows.
            throw new ConfigurationException("authorizationLocation",
            "Neither an authorization URI nor a configuration URI has been given.");
        }

        if (userLocation == null && configurationLocation == null && clientAuth != null) {
            // we need a userinfo endpoint for GUI flows.
            throw new ConfigurationException("userLocation",
            "Neither a user URI nor a configuration URI has been given.");
        }

        // A token endpoint is required for robot and interactive user cases.
        if (tokenLocation == null && configurationLocation == null) {
            throw new ConfigurationException("tokenLocation",
            "Neither a token URI nor a configuration URI has been given.");
        }

        if (log.isDebugEnabled()) {
            if (clientAuth != null) {
                log.debug("Setting bind credentials for domain [{}] to [{}].",
                          domain,clientAuth.getUserName());
            }
            if (robotAuth != null) {
                log.debug("Setting robot bind credentials for domain [{}] to [{}] with grant type [{}].",
                          domain,robotAuth.getUserName(),robotGrantType);
            }
            if (robotUserCredentials != null) {
                log.debug("Setting robot user credentials for domain [{}] to [{}].",
                          domain,robotUserCredentials.getUserName());
            }
            log.debug("Setting configuration location for domain [{}] to [{}].",domain,configurationLocation);
            log.debug("Setting authorization location for domain [{}] to [{}].",domain,authorizationLocation);
            log.debug("Setting favicon location for domain [{}] to [{}].",domain,faviconLocation);
            log.debug("Setting token location for domain [{}] to [{}].",domain,tokenLocation);
            log.debug("Setting user location for domain [{}] to [{}].",domain,userLocation);
            log.debug("Setting scope for domain [{}] to [{}].",domain,scope);
            log.debug("Setting prompt for domain [{}] to [{}].",domain,prompt);
            log.debug("Setting accessType for domain [{}] to [{}].",domain,accessType);
            log.debug("Setting resource for domain [{}] to [{}].",domain,resource);
            log.debug("Setting groupRoleResource for domain [{}] to [{}].",domain,groupRoleResource);
            log.debug("Setting options for domain [{}] to [{}].",domain,options);
            log.debug("Setting appUsers for domain [{}] to [{}].",domain,appUsers);
            log.debug("Setting robot scope for domain [{}] to [{}].",
            domain,robotScope);
        }

        return new DomainConfig(
            domain,
            label,
            authorizationLocation,
            tokenLocation,
            userLocation,
            configurationLocation,
            faviconLocation,
            clientAuth,
            scope,
            prompt,
            accessType,
            resource,
            groupRoleResource,
            options,
            appUsers,
            robotGrantType,
            robotScope,
            robotAuth,
            robotUserCredentials
        );
    }

    @Override
    public synchronized void updated(Dictionary<String,?> properties) throws ConfigurationException {
        this.stopOpenIdJobs();
        this.domainConfigurations.clear();
        this.guiDomains.clear();

        // domain -> property -> value
        Map<String, Map<String, String>> propertiesByDomain = dictToMap(properties == null ? new Hashtable<>() : properties)
            .entrySet()
            .stream()
            .flatMap(entry -> {
                String key = entry.getKey();
                if (!key.startsWith("domain.")) {
                    return Stream.empty();
                }

                if (entry.getValue() == null) {
                    return Stream.empty();
                }

                int dotIndex = key.indexOf(".", 7);

                String domain = dotIndex < 0 ? key.substring(7) : key.substring(7, dotIndex);

                String rest = key.substring(dotIndex < 0 ? 7 : dotIndex + 1);

                return Stream.of(new Triple<>(domain, rest, entry.getValue().toString()));
            })
            .collect(Collectors.groupingBy(Triple::getFirst, Collectors.toMap(Triple::getSecond, Triple::getThird)));

        for (Map.Entry<String, Map<String, String>> entry: propertiesByDomain.entrySet()) {
            String domain = entry.getKey();
            Map<String, String> domainProperties = entry.getValue();

            log.info("Extracting configuration for OAuth domain [{}]...",domain);

            if (log.isDebugEnabled()) {
                log.debug("Parsing domain properties [{}]",domainProperties);
            }

            DomainConfig config;
            try {
                config = extractConfig(domain, domainProperties);
            } catch (ConfigurationException e) {
                throw new ConfigurationException("domain." + domain + "." + e.getProperty(), e.getReason());
            }

            log.info("Final OAuth domain config is [{}].",config);

            this.domainConfigurations.put(domain,
                new DomainConfigRecord(
                    new DomainManager(
                            config, Clock.systemDefaultZone(),
                            this.oauthHttpClient, this.tokenValidator),
                    null));

            if (config.getClientCredentials() != null) {
                this.guiDomains.add(domain);
            }
        }

        this.startOpenIdJobs();
    }

    /**
     * @return the list of domains with a user flow for which we have a domain config.
     */
    public synchronized List<String> getDomains() {

        return new ArrayList<String>(this.guiDomains);
    }

    /**
     * @param domain A configured domain name.
     * @return The domain configuration for the given domain.
     */
    public synchronized DomainManager getDomainManager(String domain) {
        return Optional.ofNullable(this.domainConfigurations.get(domain))
            .map(r -> r.getDomainManager())
            .orElse(null);
    }

    public void setOauthHttpClient(OAuthHttpClient oauthHttpClient) {
        this.oauthHttpClient = oauthHttpClient;
    }

    public void setTokenValidator(TokenValidator tokenValidator) {
        this.tokenValidator = tokenValidator;
    }

    public void setSecretsService(BiFunction<String, String, String> secretsService) {
        this.secretsService = secretsService;
    }

    public void setBlueprintBundle(Bundle blueprintBundle) {
        this.blueprintBundle = blueprintBundle;
    }
}
