/***********************************************************
 * $Id$
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.07.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.util.EnumSet;

/**
 * A set of config options for individual domains.
 */
public enum ConfigOptions {

    /**
     * Whether to skip the access token hash check, if no
     * <code>at_hash</code> claim is contained in an ID Token
     * delivered together with an access token.
     */
    lenientAccessTokenCheck,

    /**
     * Whether to propagate the locale of the login service to the OAuth providers
     * authorization endpoint using a <code>locale</code> URL parameter.
     */
    propagateLocale,

    /**
     * Whether we assume, that the OAuth provider is seen by the clients
     * under the same hostname than our OAuth client.
     *
     * This is the case, when both services are integrated to the same
     * ReverseProxy setup.
     */
    reverseProxyIntegrated,

    /**
     * Send the client id and client secret using basic auth.
     * (This is the preferred way to do it according to the oauth standard)
     */
    basicAuthCredentials,

    /**
     * parse the access token rather than the ID Token for fetching the identity.
     */
    jwtAccessToken,

    /**
     * Render a user logout button in order to logout from the target system
     * by redirecting the browser to the logout URI.
     */
    renderUserLogout,
    ;


    /**
     * Parse a string of comma-separated configuration options.
     *
     * @param opts A comma-separated list of options.
     * @return The parsed set of options contained in the comma-separated string.
     */
    public static EnumSet<ConfigOptions> parseOptions(String opts) {

        EnumSet<ConfigOptions> ret = EnumSet.noneOf(ConfigOptions.class);

        if (opts != null) {

            String[] optsArray = opts.split(",");

            for (String opt : optsArray) {

                opt = opt.trim();

                if (opt.length() > 0) {
                    ret.add(ConfigOptions.valueOf(opt.trim()));
                }
            }
        }

        return ret;
    }
}
