/***********************************************************
 * $Id$
 * 
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.06.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.oauth;

import java.util.Locale;

/**
 * A state of an ongoing authorization process.
 */
public class AuthState {

    private final String state;
    private final Locale locale;
    private String domain;
    private long expires;
    private OAuthTokenErrorResponse error;
    private OAuthTokenResponse response;
    
    public AuthState(String state, long expires, Locale locale) {
        super();
        this.state = state;
        this.domain = null;
        this.expires = expires;
        this.locale = locale;
    }
    
    public synchronized void init(String domain1, long expires1) {
        
        if (this.domain != null) {
            throw new IllegalStateException("Cannot select domain twice.");
        }
        
        this.domain = domain1;
        this.expires = expires1;
    }

    public String getState() {
        return this.state;
    }

    public synchronized String getDomain() {
        return this.domain;
    }

    public synchronized long getExpires() {
        return this.expires;
    }
    
    public Locale getLocale() {
        return this.locale;
    }
    
    private void checkState() {

        if (this.response != null || this.error != null) {
            throw new IllegalStateException("Cannot commit state twice.");
        }
    }

    public synchronized void setError(OAuthTokenErrorResponse error) {
        
        this.checkState();
        this.error = error;
        this.notifyAll();
    }
    
    public synchronized void setResponse(OAuthTokenResponse response) {
    
        this.checkState();
        this.response = response;
        this.notifyAll();
    }
    
    public synchronized OAuthTokenResponse waitForResponse(long timeout) throws OAuthTokenErrorResponse, InterruptedException {
        
        if (this.response != null) {
            return this.response;
        }
        if (this.error != null) {
            throw this.error;
        }
        
        if (timeout > 0) {
            this.wait(timeout);

            if (this.response != null) {
                return this.response;
            }
            if (this.error != null) {
                throw this.error;
            }
        }
        
        return null;
    }

}
