/* **********************************************************
 * $Id$
 *
 * http://www.clazzes.org
 *
 * Created: 02.04.2011
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.ldap;

import java.net.PasswordAuthentication;
import java.net.URI;
import java.util.List;
import java.util.Vector;

import javax.naming.InvalidNameException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.*;
import javax.naming.ldap.*;

import org.clazzes.util.sec.DomainGroup;
import org.clazzes.util.sec.DomainPasswordLoginService;
import org.clazzes.util.sec.DomainPrincipal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LdapDomainPasswordLoginService implements
        DomainPasswordLoginService {

    private static final Logger log = LoggerFactory.getLogger(LdapDomainPasswordLoginService.class);

    private ConfigurationService configurationService;
    private GroupInfoCache groupInfoCache;


    @Override
    public DomainPrincipal tryLogin(String domain, String username, String password) {

        LdapContext searchCtxt = null;
        ClassLoader ocl = Thread.currentThread().getContextClassLoader();

        try {

            Thread.currentThread().setContextClassLoader(ClassLoader.getSystemClassLoader());

            DomainConfig domainConfig = this.configurationService.getDomainController(domain);
            if (domainConfig == null)
                throw new SecurityException("Invalid domain ["+domain+"] specified.");

            if (password.isEmpty() && !domainConfig.isAllowEmptyPasswords()) {

                throw new SecurityException("Domain ["+domain+"] does not allow empty passwords to be specified.");
            }

            URI uri = AdsHelper.resolveServerURI(domainConfig.getControllerUri());

            if (log.isDebugEnabled())
                log.debug("Connecting to LDAP server [{}] for domain [{}]...",uri,domain);

            AdsPrincipal userPrincipal = null;
            String dn;

            if (ConfigurationService.AUTHMETHOD_BINDADS.equals(domainConfig.getAuthMethod())) {

                dn = username+"@"+domain;
                userPrincipal = new AdsPrincipal(username, domain, username, null);
            }
            else if (ConfigurationService.AUTHMETHOD_SEARCH_AND_BIND.equals(domainConfig.getAuthMethod())) {

                searchCtxt = AdsHelper.connectToADS(uri,domainConfig.getBindCredentials(),domainConfig.getAuthMechanism());

                if (log.isDebugEnabled())
                    log.debug("Sucessfully connected to LDAP server [{}] for domain [{}].",uri,domain);

                SearchControls ctrl = new SearchControls();
                ctrl.setSearchScope(SearchControls.SUBTREE_SCOPE);

                String filter = "{0}={1}";

                if (log.isDebugEnabled())
                    log.debug("Searching [{}={}]...",domainConfig.getUserAttribute(),username);

                NamingEnumeration<SearchResult> searchResults = searchCtxt.search(
                        domainConfig.getBaseDnToUsers(),
                        filter,
                        new Object[]{domainConfig.getUserAttribute(),username},
                        ctrl);

                if (!searchResults.hasMoreElements())
                    throw new SecurityException("User ["+username+"] not found in domain ["+domain+"].");

                SearchResult result = searchResults.next();
                dn = AdsHelper.getAbsoluteDn(uri, domainConfig.getBaseDnToUsers(), result);

                log.info("User [{}] in domain [{}] resolved to DN [{}].",new Object[]{username,domain,dn});

                userPrincipal = AdsHelper.createPrincipal(domainConfig, result);

                searchCtxt.close();
                searchCtxt = null;
            }
            else {
                throw new SecurityException("Invalid authentication method ["+domainConfig.getAuthMethod()+"] specified.");
            }

            PasswordAuthentication auth = new PasswordAuthentication(dn,password.toCharArray());

            if (log.isDebugEnabled())
                log.debug("Binding as [{}] to domain [{}]...",dn,domain);

            searchCtxt = AdsHelper.connectToADS(uri,auth,domainConfig.getAuthMechanism());

            return userPrincipal;

        } catch (Exception e) {

            if (log.isDebugEnabled())
                log.debug("Caught exception during LDAP authentication",e);

            log.error("Invalid initial login of user [{}] to domain [{}].",username,domain);
            return null;
        }
        finally {
            if (searchCtxt != null) {
                try {
                    searchCtxt.close();
                } catch (NamingException e) {
                    log.warn("Error closing LDAP context",e);
                }
            }

            if (ocl != null) {
                Thread.currentThread().setContextClassLoader(ocl);
            }
        }
    }


    @Override
    public String getDefaultDomain() {
        return this.configurationService.getDefaultDomain();
    }

    @Override
    public List<String> getDomains() {
        return this.configurationService.getDomains();
    }


    @Override
    public void changePassword(String domain, String username, String oldPassword, String newPassword) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void deactivateUser(String domain, String username, String reason) {
        throw new UnsupportedOperationException();
    }

    @Override
    public List<DomainGroup> getGroups(String domain, String userName) {

        LdapContext searchCtxt = null;
        ClassLoader ocl = Thread.currentThread().getContextClassLoader();

        // group info bucket, that is waiting for results.
        GroupInfo pendingGroupInfo = null;
        
        // fallback value for retaining error results.
        long groupTimeoutSeconds = 30L;
                
        try {

            Thread.currentThread().setContextClassLoader(ClassLoader.getSystemClassLoader());

            DomainConfig domainConfig = getCorrectDomainConfig(domain);

            long groupCacheSeconds = domainConfig.getGroupCacheSeconds();
            groupTimeoutSeconds = domainConfig.getGroupTimeoutSeconds();
            
            long now = System.currentTimeMillis();
            
            if (groupCacheSeconds > 0L) {
                
                GroupInfo groupInfo = this.groupInfoCache.getGroupInfo(domain,userName,now+1000L*groupCacheSeconds);
                
                // this method return non-null or throws, if a concurrent request failed.
                // null is returned, if the bucket is waiting for a result.
                List<DomainGroup> groups = groupInfo.waitForResult(groupTimeoutSeconds*1000L);
                
                if (groups != null) {
                    // we got a cache hit
                    if (log.isDebugEnabled()) {
                        log.debug("Returning cached groups [{}] for user [{}/{}].",groups,domain,userName);
                    }
                    return groups;
                }
                
                // from here on we are the owner of the cache bucket and are responsible for filling the result.
                pendingGroupInfo = groupInfo;
            }
            
            URI uri = AdsHelper.resolveServerURI(domainConfig.getControllerUri());

            searchCtxt = AdsHelper.connectToADS(uri,domainConfig.getBindCredentials(),domainConfig.getAuthMechanism());

            if (log.isDebugEnabled())
                log.debug("Sucessfully connected to LDAP server [{}] for domain [{}].",uri,domain);

            // 1. search the user
            if (log.isDebugEnabled())
                log.debug("Searching [{}={}]...",domainConfig.getUserAttribute(),userName);

            SearchResult result = searchForUser(domain, userName, searchCtxt, domainConfig);

            String userDn = AdsHelper.getAbsoluteDn(uri, domainConfig.getBaseDnToUsers(), result);
            log.info("User [{}] found in domain [{}] resolved to DN [{}].",new Object[]{userName,domain,userDn});

            Vector<DomainGroup> groups;
            
            // Get the (LDAP-) attributes of this user
            Attributes userAttributes = result.getAttributes();
            if (userAttributes == null) {
                log.warn("Search result with DN ["+ userDn +"] has no attributes");

                // perhaps we should throw a SecurityException at this point instead
                // However, this was not done in historic code - not changing the behavior
                groups = new Vector<DomainGroup>();
            }
            else {
                // 1b look whether the user was disabled
                if (!domainConfig.isAllowGroupsForDisabledUser() && AdsHelper.isDisabledAdUser(userAttributes)) {
                    throw new SecurityException("User account " + domain + "\\" + userName + " was disabled.");
                }
                
                // 2. collect membership attributes
                Vector<String> groupDns = getMemberOfDns(userAttributes);
                
                // 3. collect DomainGroups with prettyName etc.
                groups = getDomainGroups(searchCtxt, domainConfig, uri, groupDns);
            }
            
            if (pendingGroupInfo != null) {
                
                if (log.isDebugEnabled()) {
                    log.debug("Caching groups [{}] for user [{}/{}].",groups,domain,userName);
                }
                
                // inform concurrent threads about the result of the group query.
                pendingGroupInfo.setResult(groups);
            }

            if (log.isDebugEnabled()) {
                log.debug("Returning groups [{}] for user [{}/{}].",groups,domain,userName);
            }
            
            return groups;

        } catch (Throwable e) {

            if (log.isDebugEnabled())
                log.debug("Caught exception during LDAP authentication",e);

            if (pendingGroupInfo != null) {
                // cache errors for the double wait timeout.
                long now = System.currentTimeMillis();
                pendingGroupInfo.setError(e.getMessage(), now + groupTimeoutSeconds * 1000L * 2L);
            }
            
            log.error("Error querying groups of user ["+domain+"/"+userName+"].",e);
            return null;
        }
        finally {
            if (searchCtxt != null) {
                try {
                    searchCtxt.close();
                } catch (NamingException e) {
                    log.warn("Error closing LDAP context",e);
                }
            }

            if (ocl != null) {
                Thread.currentThread().setContextClassLoader(ocl);
            }
        }

    }

    private SearchResult searchForUser(String domain, String userName, LdapContext searchCtxt, DomainConfig domainConfig) throws NamingException {
        String filter = "{0}={1}";
        SearchControls ctrl = new SearchControls();
        ctrl.setSearchScope(SearchControls.SUBTREE_SCOPE);

        NamingEnumeration<SearchResult> searchResults = searchCtxt.search(
                domainConfig.getBaseDnToUsers(),
                filter,
                new Object[]{domainConfig.getUserAttribute(), userName},
                ctrl);

        if (!searchResults.hasMoreElements())
            throw new SecurityException("User ["+ userName +"] not found in domain ["+ domain +"].");

        SearchResult result = searchResults.next();
        return result;
    }

    private Vector<String> getMemberOfDns(Attributes userAttributes) throws NamingException {
        Vector<String> groupDns = new Vector<String>();

        Attribute memberOfAttribute = userAttributes.get("memberOf");
        if (memberOfAttribute != null) {
            NamingEnumeration<?> memberOfAttributes = memberOfAttribute.getAll();
            while (memberOfAttributes.hasMoreElements()) {
                // we actually are getting a String anyway
                String groupDn = memberOfAttributes.nextElement().toString();
                if (log.isTraceEnabled()) {
                    // important: this code could throw Exceptions only if trace is enabled
                    String groupName = this.getGroupNameOffCn(groupDn); // pretty name, but we do not get samAccountName by this
                    log.trace("Found memberOf element with groupDn [" + groupDn + "] and CN [" + groupName + "]");
                }
                groupDns.add(groupDn);
            }
        }
        return groupDns;
    }

    private Vector<DomainGroup> getDomainGroups(LdapContext searchCtxt, DomainConfig domainConfig, URI uri, Vector<String> groupDns) throws NamingException {
        Vector<DomainGroup> userGroups = new Vector<DomainGroup>();

        SearchControls ctrl = new SearchControls();
        ctrl.setSearchScope(SearchControls.SUBTREE_SCOPE);

        if (groupDns.size() > 0) {
            for (String groupDn : groupDns) {

                String groupFilter = "distinguishedName={0}";

                if (log.isDebugEnabled())
                    log.debug("Searching [{}] with args {}", new Object[]{groupFilter, groupDn});

                String relativeDn = AdsHelper.getRelativeDn(uri, "", groupDn); // empty string is valid!

                // here, search could be Object instead of Subtree
                NamingEnumeration<SearchResult> groupResults = searchCtxt.search(
                        relativeDn,
                        groupFilter,
                        new Object[]{groupDn},
                        ctrl);

                while (groupResults.hasMoreElements()) {
                    SearchResult groupResult = groupResults.next();
                    if (groupResult == null) {
                        break;
                    }
                    AdsGroup adsGroup = AdsHelper.createGroup(domainConfig, groupResult);

                    if (log.isTraceEnabled())
                        log.trace("Found group object: [{}]", adsGroup.toString());

                    userGroups.add(adsGroup);
                }
            }
            debugLogFoundGroups(userGroups);
        }
        return userGroups;
    }

    private void debugLogFoundGroups(Vector<DomainGroup> userGroups) {
        if (log.isDebugEnabled()) {
            StringBuilder groupNames = new StringBuilder();
            for(DomainGroup group : userGroups) {
                if (groupNames.length() != 0) {
                    groupNames.append(", ");
                }
                groupNames
                        .append(group.getDomain())
                        .append("\\")
                        .append(group.getGroupName());
            }

            log.debug("Found user groups: [" + groupNames + "]");
        }
    }

    private String getGroupNameOffCn(String groupDn) throws InvalidNameException {
        LdapName groupDnName = new LdapName(groupDn);
        for(Rdn rdn : groupDnName.getRdns()) {
            if ("CN".equals(rdn.getType())) {
                Object cnValue = rdn.getValue();
                if (cnValue == null) {
                    return null;
                }

                // return the first occurence of a CN
                return cnValue.toString();
            }
        }

        // CN not found
        return null;
    }

    private DomainConfig getCorrectDomainConfig(String domain) {
        DomainConfig domainConfig = this.configurationService.getDomainController(domain);
        if (domainConfig == null) {
            throw new SecurityException("Invalid domain [" + domain + "] specified.");
        }

        if (domainConfig.getBindCredentials() == null ||
                domainConfig.getBindCredentials().getUserName() == null ||
                domainConfig.getBindCredentials().getUserName().length() == 0) {
            throw new SecurityException("Domain [" + domain + "] has no bind dn.");
        }
        return domainConfig;
    }


    @Override
    public List<DomainPrincipal> getGroupMembers(String domain, String groupName) {

        LdapContext searchCtxt = null;
        ClassLoader ocl = Thread.currentThread().getContextClassLoader();

        try {

            Thread.currentThread().setContextClassLoader(ClassLoader.getSystemClassLoader());

            DomainConfig domainConfig = getCorrectDomainConfig(domain);

            URI uri = AdsHelper.resolveServerURI(domainConfig.getControllerUri());

            searchCtxt = AdsHelper.connectToADS(uri,domainConfig.getBindCredentials(),domainConfig.getAuthMechanism());

            if (log.isDebugEnabled())
                log.debug("Sucessfully connected to LDAP server [{}] for domain [{}].",uri,domain);

            SearchControls ctrl = new SearchControls();
            ctrl.setSearchScope(SearchControls.SUBTREE_SCOPE);

            // 1. search the group to detect the full DN

            //String groupFilter = "(&(objectCategory=Group)({0}={1}))";
            String groupFilter = "{0}={1}";

            if (log.isDebugEnabled())
                log.debug("Searching group [{}] in domain [{}]", new Object[]{groupName,domain});

            NamingEnumeration<SearchResult> groupSearchResults = searchCtxt.search(
                    domainConfig.getBaseDnToGroups(),
                    groupFilter,
                    new Object[]{domainConfig.getGroupAttribute(),groupName},
                    ctrl);

            if (!groupSearchResults.hasMoreElements()) {
                throw new SecurityException("Group ["+groupName+"] not found in domain ["+domain+"].");
            }

            SearchResult groupResult = groupSearchResults.next();
            String groupDN = AdsHelper.getAbsoluteDn(uri, domainConfig.getBaseDnToGroups(), groupResult);
            log.debug("Group ["+groupName+"] has full DN ["+groupDN+"].");

            // 2. search users who are member of this group
            //String memberFilter = "(&(|(objectCategory=User)(objectCategory=Group))(memberOf={0}))";
            //String memberFilter = "(&(objectCategory=User)(memberOf={0}))";
            String memberFilter = "memberOf={0}";

            if (log.isDebugEnabled())
                log.debug("Searching [{}] with args {}", new Object[]{memberFilter, groupDN});

            NamingEnumeration<SearchResult> searchResults = searchCtxt.search(
                    domainConfig.getBaseDnToUsers(), // search in user's memberOf
                    memberFilter,
                    new Object[]{groupDN},
                    ctrl);

            Vector<DomainPrincipal> memberPrincipals = new Vector<DomainPrincipal>();
            while (searchResults.hasMoreElements()) {
                SearchResult memberResult = searchResults.next();
                if (memberResult == null) {
                    break;
                }

                AdsPrincipal memberPrincipal = AdsHelper.createPrincipal(domainConfig, memberResult);

                log.debug("Found group member: [{}]", memberPrincipal.toString());

                memberPrincipals.add(memberPrincipal);

            }
            // log.info
            if (memberPrincipals.size() == 0) {
                log.warn("Group ["+groupName+"] found in domain ["+domain+"] does not exist or has no members.");
            } else {
                log.debug("Group ["+groupName+"] found in domain ["+domain+"] has "+memberPrincipals.size()+" members.");
            }

            searchCtxt.close();
            searchCtxt = null;

            return memberPrincipals;

        } catch (Exception e) {

            if (log.isDebugEnabled())
                log.debug("Caught exception during LDAP authentication",e);

            //log.error("Invalid initial login of user [{}] to domain [{}].",username,domain);
            return null;
        }
        finally {
            if (searchCtxt != null) {
                try {
                    searchCtxt.close();
                } catch (NamingException e) {
                    log.warn("Error closing LDAP context",e);
                }
            }

            if (ocl != null) {
                Thread.currentThread().setContextClassLoader(ocl);
            }
        }

    }


    @Override
    public int getSupportedFeatures(String domain) {
        return FEATURE_SEARCH_USER | FEATURE_GET_GROUPMEMBERS | FEATURE_GET_GROUPS;
    }


    @Override
    public DomainPrincipal searchUser(String domain, String userName) {

        LdapContext searchCtxt = null;
        ClassLoader ocl = Thread.currentThread().getContextClassLoader();

        try {

            Thread.currentThread().setContextClassLoader(ClassLoader.getSystemClassLoader());

            DomainConfig domainConfig = getCorrectDomainConfig(domain);

            URI uri = AdsHelper.resolveServerURI(domainConfig.getControllerUri());

            searchCtxt = AdsHelper.connectToADS(uri,domainConfig.getBindCredentials(),domainConfig.getAuthMechanism());

            if (log.isDebugEnabled())
                log.debug("Sucessfully connected to LDAP server [{}] for domain [{}].",uri,domain);

            SearchControls ctrl = new SearchControls();
            ctrl.setSearchScope(SearchControls.SUBTREE_SCOPE);

            String filter = "{0}={1}";

            if (log.isDebugEnabled())
                log.debug("Searching [{}={}]...",domainConfig.getUserAttribute(),userName);

            NamingEnumeration<SearchResult> searchResults = searchCtxt.search(
                    domainConfig.getBaseDnToUsers(),
                    filter,
                    new Object[]{domainConfig.getUserAttribute(),userName},
                    ctrl);

            if (!searchResults.hasMoreElements())
                throw new SecurityException("User ["+userName+"] not found in domain ["+domain+"].");

            SearchResult result = searchResults.next();

            String userDn = AdsHelper.getAbsoluteDn(uri, domainConfig.getBaseDnToUsers(), result);
            log.info("User [{}] found in domain [{}] resolved to DN [{}].",new Object[]{userName,domain,userDn});

            AdsPrincipal userPrincipal = AdsHelper.createPrincipal(domainConfig, result);

            searchCtxt.close();
            searchCtxt = null;

            return userPrincipal;

        } catch (Exception e) {

            if (log.isDebugEnabled())
                log.debug("Caught exception during LDAP authentication",e);

            //log.error("Invalid initial login of user [{}] to domain [{}].",username,domain);
            return null;
        }
        finally {
            if (searchCtxt != null) {
                try {
                    searchCtxt.close();
                } catch (NamingException e) {
                    log.warn("Error closing LDAP context",e);
                }
            }

            if (ocl != null) {
                Thread.currentThread().setContextClassLoader(ocl);
            }
        }
    }


    @Override
    public void sendPassword(String domain, String username) {
        throw new UnsupportedOperationException();
    }


    public ConfigurationService getConfigurationService() {
        return this.configurationService;
    }

    public synchronized void setGroupInfoCache(GroupInfoCache groupInfoCache) {
        this.groupInfoCache = groupInfoCache;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

}
