/***********************************************************
 * $Id$
 * 
 * http://www.clazzes.org
 *
 * Created: 02.04.2011
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.ldap;

import java.net.PasswordAuthentication;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.function.BiFunction;

import org.osgi.service.blueprint.container.ServiceUnavailableException;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.cm.ManagedService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationService implements ManagedService {

    private static final Logger log = LoggerFactory.getLogger(ConfigurationService.class);

    public static final String CONFIG_PID = "org.clazzes.login.ldap";
    
    public static final String AUTHMETHOD_BINDADS = "bindAds";
    public static final String AUTHMETHOD_SEARCH_AND_BIND = "searchAndBind";

    private static final String DEFAULT_USER_ATTRIBUTE = "samAccountName";
    private static final String DEFAULT_PRETTYNAME_ATTRIBUTE = "cn";
    private static final String DEFAULT_EMAILADDRESS_ATTRIBUTE = "mail";
    private static final String DEFAULT_ALLOW_EMPTY_PASSWORDS = Boolean.toString(false);
    private static final String DEFAULT_ALLOW_GROUPS_FOR_DISABLED = Boolean.toString(true); // default true as it was historic behavior
    private static final String DEFAULT_AUTH_MECHANISM = "simple";

    private String defaultDomain;

    private BiFunction<String,String,String> secretsService;
    
    /**
     * A mapping from Windows domain names to domain controller URIs.
     */
    private final Map<String,DomainConfig> domainControllers;


    public ConfigurationService() {
        this.defaultDomain = null;
        this.domainControllers = new HashMap<String,DomainConfig>();
    }


    @Override
    public synchronized void updated(Dictionary<String,?> properties) throws ConfigurationException {

        Object s = properties == null ? null : properties.get("defaultDomain");
        if (s != null) {
            this.defaultDomain = s.toString();
            if (log.isDebugEnabled())
                log.debug("Setting default domain to [{}].",this.defaultDomain);
        }
        else {
            this.defaultDomain = null;
            if (log.isDebugEnabled())
                log.debug("Setting default domain to default [{}].",this.defaultDomain);
        }

        this.domainControllers.clear();
        
        if (properties != null) {
            Enumeration<String> keys = properties.keys();

            while (keys.hasMoreElements()) {

                String key = keys.nextElement();

                if (key.startsWith("domain.") && key.endsWith(".controllerUri")) {

                    String domain = key.substring(7,key.length()-14);
                    String uri_s = properties.get(key).toString();

                    try {

                        URI uri = new URI(uri_s);

                        if (log.isDebugEnabled())
                            log.debug("Setting controller for domain [{}] to [{}].",domain,uri);

                        Object binduser = properties.get("domain."+domain+".bindUser");
                        PasswordAuthentication auth = null;

                        if (binduser != null) {

                            Object bindpw_o = properties.get("domain."+domain+".bindPassword");
                            
                            String bindpw = bindpw_o == null ? "" : bindpw_o.toString();

                            if (log.isDebugEnabled())
                                log.debug("Setting bind credentials for domain [{}] to [{}].",domain,binduser);

                            if (bindpw.startsWith("secret::")) {
                                
                                String skey = bindpw.substring(8);
                                
                                try {
                                    bindpw = this.secretsService.apply(CONFIG_PID,skey);
                                    log.info("Resolved password secret [{}] from OSGi secrets service.",skey);
                                }
                                catch (ServiceUnavailableException e) {
                                    log.warn("Cannot resolve password secret with no secrets service available.");
                                }

                            }
                            
                            auth = new PasswordAuthentication(binduser.toString(),bindpw.toString().toCharArray());
                        }

                        // this typo has gone to the wild with version ldap-login-service-1.5.0,
                        // so interpret autMethod too, because admins might already have been adapting their configs.
                        String autMethod = getPropOrDefault(
                                properties, domain, "autMethod", "searchAndBind");
                        
                        String authMethod = getPropOrDefault(
                                properties, domain, "authMethod", autMethod);
                        
                        String userAttribute = getPropOrDefault(
                                properties, domain, "userAttribute", DEFAULT_USER_ATTRIBUTE);
                        // as historically the same property was used, the default remains userAttribute
                        String groupAttribute = getPropOrDefault(
                                properties, domain, "groupAttribute", userAttribute);
                        String prettyNameAttribute = getPropOrDefault(
                                properties, domain, "prettyNameAttribute", DEFAULT_PRETTYNAME_ATTRIBUTE);
                        String eMailAddressAttribute = getPropOrDefault(
                                properties, domain, "eMailAddressAttribute", DEFAULT_EMAILADDRESS_ATTRIBUTE);
                        String mobileAttribute = getPropOrDefault(
                                properties, domain, "mobileAttribute", null);
                        String tokenIdsAttribute = getPropOrDefault(
                                properties, domain, "tokenIdsAttribute", null);
                        String allowEmptyPasswords = getPropOrDefault(
                                properties, domain, "allowEmptyPasswords", DEFAULT_ALLOW_EMPTY_PASSWORDS);
                        String allowGroupsForDisabledUser = getPropOrDefault(
                                properties, domain, "allowGroupsForDisabledUser", DEFAULT_ALLOW_GROUPS_FOR_DISABLED);
                        String authMechanism = getPropOrDefault(
                                properties, domain, "authMechanism", DEFAULT_AUTH_MECHANISM);

                        String baseDnToUsers = getPropOrDefault(
                                properties, domain, "baseDnToUsers", "");
                        String baseDnToGroups = getPropOrDefault(
                                properties, domain, "baseDnToGroups", "");

                        long groupCacheSeconds = getPropOrDefault(
                                properties, domain, "groupCacheSeconds", 300L);
                        long groupTimeoutSeconds = getPropOrDefault(
                                properties, domain, "groupTimeoutSeconds", 30L);
                        
                        this.domainControllers.put(domain,new DomainConfig(
                                domain,
                                uri,
                                auth,
                                authMethod,
                                userAttribute,
                                groupAttribute,
                                prettyNameAttribute,
                                eMailAddressAttribute,
                                mobileAttribute,
                                tokenIdsAttribute,
                                Boolean.parseBoolean(allowEmptyPasswords),
                                Boolean.parseBoolean(allowGroupsForDisabledUser),
                                authMechanism,
                                baseDnToUsers,
                                baseDnToGroups,
                                groupCacheSeconds,
                                groupTimeoutSeconds));

                    } catch (URISyntaxException e) {
                        throw new ConfigurationException("domainControllers",
                                "Invalid format of domain controller URI ["+uri_s+"]: Invalid URI syntax: "+e.getMessage());
                    }
                }
            }
        }

    }

    private String getPropOrDefault(Dictionary<String, ?> properties, String domain, String what, String defaultValue) {
        Object configuredValue = properties.get("domain."+domain+"."+what);
        if (configuredValue == null) {
            if (log.isDebugEnabled())
                log.debug("Setting {} for domain [{}] to default [{}].", new Object[] { what, domain, defaultValue });
            return defaultValue;
        }
        if (log.isDebugEnabled())
            log.debug("Setting {} for domain [{}] to [{}].", new Object[] { what, domain, configuredValue });
        return configuredValue.toString();
    }

    private long getPropOrDefault(Dictionary<String, ?> properties, String domain, String what, long defaultValue) {
        Object configuredValue = properties.get("domain."+domain+"."+what);
        if (configuredValue == null) {
            if (log.isDebugEnabled())
                log.debug("Setting {} for domain [{}] to default [{}].", new Object[] { what, domain, defaultValue });
            return defaultValue;
        }
        
        if (log.isDebugEnabled()) {
            log.debug("Setting {} for domain [{}] to [{}].", new Object[] { what, domain, configuredValue });
        }
        
        return Long.parseLong(configuredValue.toString());
    }

    public synchronized String getDefaultDomain() {
        return this.defaultDomain;
    }


    /**
     * @return the list of domains for which we have a domain config.
     */
    public synchronized List<String> getDomains() {
        Vector<String> domains = new Vector<String>(this.domainControllers.size());
        domains.addAll(this.domainControllers.keySet());
        return domains;
    }


    public synchronized void setDefaultDomain(String defaultDomain) {
        this.defaultDomain = defaultDomain;
    }


    /**
     * @param domain A configured domain name.
     * @return The domain controller URI for the given domain.
     */
    public synchronized DomainConfig getDomainController(String domain) {

        return this.domainControllers.get(domain);
    }

    public void setSecretsService(BiFunction<String, String, String> secretsService) {
        this.secretsService = secretsService;
    }

}
