/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.06.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f.json;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.HexFormat;

import org.clazzes.login.jbo.json.JsonHelper;
import org.clazzes.login.jbo.u2f.DeviceRegistry;
import org.clazzes.login.jbo.u2f.DeviceSelector;
import org.clazzes.login.jbo.u2f.X509DeviceSelector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;

/**
 * Parser for Yubico device registries.
 */
public abstract class DeviceRegistryParser {

    private static final Logger log = LoggerFactory.getLogger(DeviceRegistryParser.class);

    private static final CertificateFactory certificateFactory;
    private static final SimpleModule customModule;

    static {

        customModule = new SimpleModule();
        customModule.addDeserializer(X509Certificate.class,new CertificateDeserializer());
        customModule.addDeserializer(DeviceSelector.class,new DeviceSelectorDeserializer());

        try {
            certificateFactory = CertificateFactory.getInstance("X.509");
        } catch (CertificateException e) {
            throw new RuntimeException("X.509 CertificateFactory could not be instantiated.",e);
        }
    }

    private static class CertificateDeserializer extends JsonDeserializer<X509Certificate> {

        @Override
        public X509Certificate deserialize(JsonParser p, DeserializationContext ctxt)
                throws IOException, JacksonException {

            String s = p.getText();

            try {
                return (X509Certificate) certificateFactory.generateCertificate(new ByteArrayInputStream(s.getBytes("US-ASCII")));
            } catch (CertificateException e) {
                throw new IOException("Unable to parse certificate from JSON attribute.",e);
            }
        }
    }

    private static class DeviceSelectorDeserializer extends JsonDeserializer<DeviceSelector> {

        @Override
        public DeviceSelector deserialize(JsonParser p, DeserializationContext dctxt)
                throws IOException, JacksonException {

            String ctxt = "DeviceSelector";

            JsonHelper.beginObject(p,ctxt);

            String key = JsonHelper.nextName(p,ctxt);
            if (!"type".equals(key)) {
                throw new IllegalArgumentException("Device Selector does not start with a [type] attribute.");
            }

            String type = JsonHelper.nextString(p,ctxt);

            if (!"x509Extension".equals(type)) {
                throw new IllegalArgumentException("Device Selector has a type different from [x509Extension].");
            }

            key = JsonHelper.nextName(p,ctxt);
            if (!"parameters".equals(key)) {
                throw new IllegalArgumentException("Device Selector does not contain a [parameters] attribute.");
            }

            JsonHelper.beginObject(p,ctxt);

            String oid = null;
            Object value = null;

            while ((key = JsonHelper.nextName(p,ctxt)) != null) {

                if ("key".equals(key)) {
                    oid = JsonHelper.nextString(p,ctxt);
                }
                else if ("value".equals(key)) {

                    JsonToken token = p.nextToken();

                    if (token == JsonToken.START_OBJECT) {

                        key = JsonHelper.nextName(p,ctxt);
                        if (!"type".equals(key)) {
                            throw new IllegalArgumentException("X509Extension Device Selector value does not start with a [type] attribute.");
                        }

                        type = JsonHelper.nextString(p,ctxt);

                        if (!"hex".equals(type)) {
                            throw new IllegalArgumentException("X509Extension Device Selector value has a type different from [hex].");
                        }

                        key = JsonHelper.nextName(p,ctxt);
                        if (!"value".equals(key)) {
                            throw new IllegalArgumentException("X509Extension Device Selector value does not contain a [value] attribute.");
                        }

                        value = HexFormat.of().parseHex(JsonHelper.nextString(p,ctxt));

                        p.nextToken();
                        JsonHelper.endObject(p,ctxt);
                    }
                    else if (token == JsonToken.VALUE_NUMBER_INT) {
                        value = p.getIntValue();
                    }
                    else {
                        value = p.getText();
                    }
                }
                else {
                    throw new IllegalArgumentException("X509Extension Device Selector contains unknown attribute ["+key+"].");
                }
            }

            // end of parameters
            JsonHelper.endObject(p,ctxt);

            // end of device selector.
            p.nextToken();
            JsonHelper.endObject(p,ctxt);

            if (oid == null) {
                throw new IllegalArgumentException("X509Extension Device Selector does not contain a [key] attribute.");
            }

            if (log.isDebugEnabled()) {
                log.debug("Parsed X509DeviceSelector: oid={},value={}",oid,value);
            }

            return new X509DeviceSelector(oid,value);
        }

    }

    public static final DeviceRegistry parseJson(InputStream is) throws IOException {

        ObjectMapper mapper = new ObjectMapper();
        mapper.registerModule(customModule);

        try (InputStreamReader reader = new InputStreamReader(is, "UTF-8")) {

            return mapper.readValue(reader,DeviceRegistry.class);
        }
    }
}
