/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 05.06.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f.json;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;

import org.clazzes.login.jbo.common.Helpers;
import org.clazzes.login.jbo.json.JsonHelper;
import org.clazzes.login.jbo.u2f.AuthenticatorAssertionResponse;
import org.clazzes.login.jbo.u2f.AuthenticatorAttestationResponse;
import org.clazzes.login.jbo.u2f.ClientData;
import org.clazzes.login.jbo.u2f.PublicKeyCredential;
import org.clazzes.login.jbo.u2f.SafetyNetPayload;
import org.clazzes.login.jbo.u2f.SafetyNetSignatureHeader;

import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;

public abstract class CredentialParser {

    private static final SimpleModule commonModule;
    private static final SimpleModule attestationModule;
    private static final SimpleModule assertionModule;
    private static final CertificateFactory certificateFactory;

    static {
        try {
            certificateFactory = CertificateFactory.getInstance("X.509");
        } catch (CertificateException e) {
            throw new RuntimeException("X.509 CertificateFactory could not be instantiated.",e);
        }

        commonModule = new SimpleModule();

        commonModule.addDeserializer(X509Certificate.class,new B64CertificateDeserializer());
        commonModule.addDeserializer(byte[].class,new B64ByteArrayDeserializer());

        attestationModule = new SimpleModule();
        attestationModule.addDeserializer(byte[].class,new B64ByteArrayDeserializer());
        attestationModule.addDeserializer(PublicKeyCredential.class,new AttestationPublicKeyCredentialDeserializer());

        assertionModule = new SimpleModule();
        assertionModule.addDeserializer(byte[].class,new B64ByteArrayDeserializer());
        assertionModule.addDeserializer(PublicKeyCredential.class,new AssertionPublicKeyCredentialDeserializer());
    }

    private static class B64ByteArrayDeserializer extends JsonDeserializer<byte[]> {

        @Override
        public byte[] deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JacksonException {

            String b64 = p.getText();

            return Helpers.parseBase64(b64);
        }
    }

    private static class B64CertificateDeserializer extends JsonDeserializer<X509Certificate> {

        @Override
        public X509Certificate deserialize(JsonParser p, DeserializationContext ctxt)
                throws IOException, JacksonException {

            String b64 = p.getText();

            try {
                return (X509Certificate) certificateFactory.generateCertificate(new ByteArrayInputStream(Helpers.parseBase64(b64)));
            } catch (CertificateException e) {
                throw new IOException("Unable to parse certificate from JSON attribute.",e);
            }
        }
    }

    private static final class AttestationPublicKeyCredentialDeserializer extends JsonDeserializer<PublicKeyCredential> {

        @Override
        public PublicKeyCredential deserialize(JsonParser p, DeserializationContext dctxt)
                throws IOException, JacksonException {

            String ctxt = "AttestationPublicKeyCredential";

            JsonHelper.beginObject(p,ctxt);

            byte[] id = null;
            AuthenticatorAttestationResponse response = null;

            String key;

            while ((key = JsonHelper.nextName(p,ctxt)) != null) {

                if ("id".equals(key)) {
                    id = Helpers.parseBase64(JsonHelper.nextString(p,ctxt));
                }
                else if ("response".equals(key)) {

                    byte[] clientData = null;
                    byte[] attestationObject = null;

                    JsonHelper.beginObject(p,ctxt);

                    String k;

                    while ((k = JsonHelper.nextName(p,ctxt)) != null) {

                        if ("clientDataJSON".equals(k)) {
                            clientData = Helpers.parseBase64(JsonHelper.nextString(p,ctxt));
                        }
                        else if ("attestationObject".equals(k)) {
                            attestationObject = Helpers.parseBase64(JsonHelper.nextString(p,ctxt));
                        }
                        else {
                            throw new IllegalArgumentException("Unknown key ["+k+"] in AuthenticatorAttestationResponse.");
                        }
                    }

                    response = new AuthenticatorAttestationResponse(clientData,attestationObject);
                }
                else if ("clientExtensionResults".equals(key)) {

                    if (JsonHelper.beginObjectOrNull(p,ctxt)) {
                        if (JsonHelper.nextName(p,ctxt) != null) {
                            throw new IllegalArgumentException("Only empty clientExtensionResults are supported.");
                        }
                    }
                }

            }

            return new PublicKeyCredential(id,response);
        }
    };

    private static final class AssertionPublicKeyCredentialDeserializer extends JsonDeserializer<PublicKeyCredential> {

        @Override
        public PublicKeyCredential deserialize(JsonParser p, DeserializationContext dctxt)
                throws IOException, JacksonException {

            String ctxt = "AssertionPublicKeyCredential";

            JsonHelper.beginObject(p,ctxt);

            byte[] id = null;
            AuthenticatorAssertionResponse response = null;

            String key;

            while ((key = JsonHelper.nextName(p,ctxt)) != null) {

                if ("id".equals(key)) {
                    id = Helpers.parseBase64(JsonHelper.nextString(p,ctxt));
                }
                else if ("response".equals(key)) {

                    byte[] clientData = null;
                    byte[] authenticatorData = null;
                    byte[] signature = null;
                    byte[] userHandle = null;

                    JsonHelper.beginObject(p,ctxt);

                    String k;

                    while ((k = JsonHelper.nextName(p,ctxt)) != null) {

                        if ("clientDataJSON".equals(k)) {
                            clientData = Helpers.parseBase64(JsonHelper.nextString(p,ctxt));
                        }
                        else if ("authenticatorData".equals(k)) {
                            authenticatorData = Helpers.parseBase64(JsonHelper.nextString(p,ctxt));
                        }
                        else if ("signature".equals(k)) {
                            signature = Helpers.parseBase64(JsonHelper.nextString(p,ctxt));
                        }
                        else if ("userHandle".equals(k)) {
                            userHandle = Helpers.parseBase64(JsonHelper.nextString(p,ctxt));
                        }
                        else {
                            throw new IllegalArgumentException("Unknown key ["+k+"] in AuthenticatorAssertionResponse.");
                        }
                    }

                    response = new AuthenticatorAssertionResponse(clientData,authenticatorData,signature,userHandle);
                }
                else if ("clientExtensionResults".equals(key)) {

                    if (JsonHelper.beginObjectOrNull(p,ctxt)) {
                        if (JsonHelper.nextName(p,ctxt) != null) {
                            throw new IllegalArgumentException("Only empty clientExtensionResults are supported.");
                        }
                    }
                }

            }

            return new PublicKeyCredential(id,response);
        }
    };

    public static PublicKeyCredential parseAttestationCredentials(Reader json) throws IOException {

        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule(attestationModule);

        return objectMapper.readValue(json,PublicKeyCredential.class);
    }

    public static PublicKeyCredential parseAttestationCredentials(String json) throws IOException {
        try (StringReader r = new StringReader(json)) {
            return parseAttestationCredentials(r);
        }
    }

    public static PublicKeyCredential parseAttestationCredentials(InputStream is) throws IOException {
        try (InputStreamReader json = new InputStreamReader(is,"UTF-8")) {
            return parseAttestationCredentials(json);
        }
    }

    public static PublicKeyCredential parseAttestationCredentials(byte[] attestationCredentials) throws IOException {
        return parseAttestationCredentials(new ByteArrayInputStream(attestationCredentials));
    }

    public static PublicKeyCredential parseAssertionCredentials(Reader json) throws IOException {

        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule(assertionModule);

        return objectMapper.readValue(json,PublicKeyCredential.class);
    }

    public static PublicKeyCredential parseAssertionCredentials(String json) throws IOException {
        try (StringReader r = new StringReader(json)) {
            return parseAssertionCredentials(r);
        }
    }
    public static PublicKeyCredential parseAssertionCredentials(InputStream is) throws IOException {
        try (InputStreamReader json = new InputStreamReader(is,"UTF-8")) {
            return parseAssertionCredentials(json);
        }
    }

    public static PublicKeyCredential parseAssertionCredentials(byte[] assertionCredentials) throws IOException {
        return parseAssertionCredentials(new ByteArrayInputStream(assertionCredentials));
    }

    public static ClientData parseClientData(Reader json) throws IOException {

        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule(commonModule);
        return objectMapper.readValue(json,ClientData.class);
    }

    public static ClientData parseClientData(String json) throws IOException {
        try (StringReader r = new StringReader(json)) {
            return parseClientData(r);
        }
    }

    public static ClientData parseClientData(InputStream is) throws IOException {
        try (InputStreamReader json = new InputStreamReader(is,"UTF-8")) {
            return parseClientData(json);
        }
    }

    public static ClientData parseClientData(byte[] clientData) throws IOException {
        return parseClientData(new ByteArrayInputStream(clientData));
    }

    public static SafetyNetPayload parseSafetyNetPayload(Reader json) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule(commonModule);
        return objectMapper.readValue(json,SafetyNetPayload.class);
    }

    public static SafetyNetPayload parseSafetyNetPayload(String json) throws IOException {
        try (StringReader r = new StringReader(json)) {
            return parseSafetyNetPayload(r);
        }
    }

    public static SafetyNetPayload parseSafetyNetPayload(InputStream is) throws IOException {
        try (InputStreamReader json = new InputStreamReader(is,"UTF-8")) {
            return parseSafetyNetPayload(json);
        }
    }

    public static SafetyNetPayload parseSafetyNetPayload(byte[] safetyNetPayload) throws IOException {
        return parseSafetyNetPayload(new ByteArrayInputStream(safetyNetPayload));
    }

    public static SafetyNetSignatureHeader parseSafetyNetSignatureHeader(Reader json) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule(commonModule);
        return objectMapper.readValue(json,SafetyNetSignatureHeader.class);
    }

    public static SafetyNetSignatureHeader parseSafetyNetSignatureHeader(String json) throws IOException {
        try (StringReader r = new StringReader(json)) {
            return parseSafetyNetSignatureHeader(r);
        }
    }

    public static SafetyNetSignatureHeader parseSafetyNetSignatureHeader(InputStream is) throws IOException {
        try (InputStreamReader json = new InputStreamReader(is,"UTF-8")) {
            return parseSafetyNetSignatureHeader(json);
        }
    }

    public static SafetyNetSignatureHeader parseSafetyNetSignatureHeader(byte[] safetyNetPayload) throws IOException {
        return parseSafetyNetSignatureHeader(new ByteArrayInputStream(safetyNetPayload));
    }

}
