/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 12.06.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.u2f.impl;

import java.io.IOException;
import java.security.MessageDigest;
import java.security.Signature;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;

import org.clazzes.login.jbo.bc.BCTools;
import org.clazzes.login.jbo.common.CurveType;
import org.clazzes.login.jbo.common.PubKeyInfo;
import org.clazzes.login.jbo.u2f.AttestationCertInfo;
import org.clazzes.login.jbo.u2f.AttestationCertValidator;
import org.clazzes.login.jbo.u2f.AttestationObject;
import org.clazzes.login.jbo.u2f.AttestationObjectValidator;
import org.clazzes.login.jbo.u2f.AttestationResult;
import org.clazzes.login.jbo.u2f.AttestationStatement;
import org.clazzes.login.jbo.u2f.AttestationType;
import org.clazzes.login.jbo.u2f.AttestedCredentialData;
import org.clazzes.login.jbo.u2f.AuthenticatorAttestationResponse;
import org.clazzes.login.jbo.u2f.AuthenticatorData;
import org.clazzes.login.jbo.u2f.DeviceInfo;
import org.clazzes.login.jbo.u2f.DeviceRegistry;
import org.clazzes.login.jbo.u2f.FidoU2FAttestationStatement;
import org.clazzes.login.jbo.u2f.PackedAttestationStatement;
import org.clazzes.login.jbo.u2f.PublicKeyCredential;
import org.clazzes.login.jbo.u2f.SafetyNetAttestationStatement;
import org.clazzes.login.jbo.u2f.SafetyNetPayload;
import org.clazzes.login.jbo.u2f.json.DeviceRegistryParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A validator for attestation objects.
 */
public class AttestationObjectValidatorImpl implements AttestationObjectValidator {

    private static final Logger log = LoggerFactory.getLogger(AttestationObjectValidatorImpl.class);
    
    private static final DeviceRegistry safetyNetRegistry;
    private static final DeviceRegistry chromeRegistry;
    private static final String idGoogleFingerprint = "1.3.6.1.4.1.11129.999.1";
    private static final String idGoogleHardwareToken = "1.3.6.1.4.1.11129.999.2";

    static {
        try {
            safetyNetRegistry = DeviceRegistryParser.parseJson(AttestationObjectValidatorImpl.class.getResourceAsStream("safetynet-metadata.json"));
        } catch (IOException e) {
            throw new RuntimeException("SafetyNet device registry could not be instantiated.",e);
        }
        try {
            chromeRegistry = DeviceRegistryParser.parseJson(AttestationObjectValidatorImpl.class.getResourceAsStream("chrome-metadata.json"));
        } catch (IOException e) {
            throw new RuntimeException("Chrome device registry could not be instantiated.",e);
        }
    }

    private AttestationCertValidator certValidator;
    
    public void setCertValidator(AttestationCertValidator certValidator) {
        this.certValidator = certValidator;
    }

    protected AttestationResult validatePackedAttestationStatement(byte[] clientDataHash, AttestationObject attestationObject, PackedAttestationStatement pa) throws Exception {
        
        X509Certificate[] chain = pa.getCertificateChain();
        
        if (chain == null) {
            // self attestation
            AuthenticatorData authData = attestationObject.parseAuthData();
            
            PubKeyInfo pubKeyInfo = authData.getAttestedCredentialData().getCredentialPublicKey();
            
            Map<String, Object> extensionValues = Collections.singletonMap(AttestationCertInfo.idFidoGenCeAaguid,authData.getAttestedCredentialData().getAaguid());
            AttestationCertInfo aci = new AttestationCertInfo(extensionValues, pubKeyInfo);
            
            DeviceInfo di = chromeRegistry.selectDevice(aci);
            
            if (di == null) {
                throw new SecurityException("Device for self attestation ["+aci+"] could not be found.");
            }
            
            Signature sig = Signature.getInstance(pubKeyInfo.getJceAlgorithm());
            
            sig.initVerify(pubKeyInfo.getPublicKey());
            
            sig.update(attestationObject.getAuthData());
            sig.update(clientDataHash);
            
            boolean r = sig.verify(pa.getSignature());
            
            if (!r) {
                throw new SecurityException("Error verifying packed self attestation signature for ["+aci+"].");
            }
            
            return new AttestationResult(aci,chromeRegistry.getIdentifier(),di,chromeRegistry.getVendorInfo(),AttestationType.SELF);
        }
        else {
            AttestationResult result = this.certValidator.validateCertificateChain(pa.getAlgorithm(),pa.getCertificateChain());
            X509Certificate attestationCert = pa.getCertificateChain()[0];
            
            Signature sig = Signature.getInstance(result.getAttestationCertInfo().getPubKeyInfo().getJceAlgorithm());
            
            sig.initVerify(attestationCert);
            
            sig.update(attestationObject.getAuthData());
            sig.update(clientDataHash);
            
            boolean r = sig.verify(pa.getSignature());
            
            if (!r) {
                throw new SecurityException("Error verifying packed attestation statement signature for ["+result.getAttestationCertInfo()+"].");
            }
            
            return result;
        }
    }
    
    protected AttestationResult validateFidoU2FAttestationStatement(byte[] clientDataHash, AttestationObject attestationObject, FidoU2FAttestationStatement pa) throws Exception {
        
        AttestationResult result = this.certValidator.validateCertificateChain(pa.getAlgorithm(),pa.getCertificateChain());
        X509Certificate attestationCert = pa.getCertificateChain()[0];
        
        AuthenticatorData authData = attestationObject.parseAuthData();
        
        AttestedCredentialData acd = authData.getAttestedCredentialData();
        
        ECPublicKey ecKey = (ECPublicKey)acd.getCredentialPublicKey().getPublicKey();
        
        AttestationCertInfo aci = result.getAttestationCertInfo();
        
        if (aci.getPubKeyInfo().getCurve() != CurveType.P_256) {
            throw new SecurityException("fido-u2f attestation statement certificate ["+aci+"] has a public key, which is not a P-256 curve.");
        }
        
        byte[] ecEncoded = BCTools.encodePoint(aci.getPubKeyInfo().getCurve(),ecKey.getW());
        
        Signature sig = Signature.getInstance(aci.getPubKeyInfo().getJceAlgorithm());
        
        sig.initVerify(attestationCert);

        // 0x00 || rpIdHash || clientDataHash || credentialId || publicKeyU2F
        sig.update(new byte[1]);
        sig.update(authData.getRpIdHash());
        sig.update(clientDataHash);
        sig.update(acd.getCredentialId());
        sig.update(ecEncoded);
        
        boolean r = sig.verify(pa.getSignature());

        if (!r) {
            throw new SecurityException("Error verifying fido-u2f attestation statement signature for ["+aci+"].");
        }
        
        return result;
    }

    protected AttestationResult validateSafetyNetAttestationStatement(byte[] clientDataHash, AttestationObject attestationObject, SafetyNetAttestationStatement pa) throws Exception {

        this.certValidator.validateServerCertificateChain(pa.getCertificateChain(),"attest.android.com");
        
        X509Certificate attestationCert = pa.getCertificateChain()[0];
        
        AttestationCertInfo aci = this.certValidator.parseCertificate(pa.getAlgorithm(),attestationCert);

        SafetyNetPayload payload = pa.getPayload();
        
        if (log.isDebugEnabled()) {
            log.debug("SafetyNetPayload=[{}]",payload);
        }
        
        AuthenticatorData authData = attestationObject.parseAuthData();
        
        if (log.isDebugEnabled()) {
            log.debug("SafetyNet authData = [{}]",authData);
        }
        
        MessageDigest sha256 = MessageDigest.getInstance("SHA-256");
        
        sha256.update(attestationObject.getAuthData());
        sha256.update(clientDataHash);
        
        byte[] nonce = sha256.digest();
        
        if (!Arrays.equals(payload.getNonce(),nonce)) {
            throw new SecurityException("safetynet attestation statement for ["+aci+"] has invalid nonce value.");
        }
        
        Signature sig = Signature.getInstance(aci.getPubKeyInfo().getJceAlgorithm());
        
        sig.initVerify(attestationCert);

        sig.update(pa.getSignaturePayload());
        
        boolean r = sig.verify(pa.getSignature());

        if (!r) {
            throw new SecurityException("Error verifying fido-u2f attestation statement signature for ["+aci+"].");
        }
        
        String et = pa.getPayload().getEvaluationType();
        
        String id = idGoogleFingerprint;

        if (et != null) {
            String[] tokens = et.trim().split("\\s*,\\s*");
            
            // check for a hardware token.
            // https://groups.google.com/forum/#!msg/safetynet-api-clients/lpDXBNeV7Fg/Ov2H6ZvhBQAJ
            
            for (String token:tokens) {
                if ("HARDWARE_BACKED".equals(token)) {
                    id = idGoogleHardwareToken;
                }
            }
        }
        
        DeviceInfo di = safetyNetRegistry.selectDevice(id);

        return new AttestationResult(aci,safetyNetRegistry.getIdentifier(),di,safetyNetRegistry.getVendorInfo(),AttestationType.CERTIFICATE);
    }
    
    @Override
    public AttestationResult validateAttestation(byte[] clientData, AttestationObject attestationObject)
            throws Exception {
        
        MessageDigest sha256 = MessageDigest.getInstance("SHA-256");
        
        byte[] clientDataHash = sha256.digest(clientData);

        AttestationStatement attStmt = attestationObject.getAttStmt();
        String fmt = attStmt.getFormat();
        
        AttestationResult result;
        
        if (PackedAttestationStatement.FORMAT.contentEquals(fmt)) {
            
            result = this.validatePackedAttestationStatement(clientDataHash,attestationObject,(PackedAttestationStatement)attStmt);
        }
        else if (FidoU2FAttestationStatement.FORMAT.contentEquals(fmt)) {
         
            result = this.validateFidoU2FAttestationStatement(clientDataHash,attestationObject,(FidoU2FAttestationStatement) attStmt);
        }
        else if (SafetyNetAttestationStatement.FORMAT.contentEquals(fmt)) {
            
            result = this.validateSafetyNetAttestationStatement(clientDataHash,attestationObject,(SafetyNetAttestationStatement) attStmt);
        }
        else {
            throw new SecurityException("Unsupported attestation format ["+fmt+"].");
        }
        
        return result;
    }

    @Override
    public AttestationResult validateAttestation(PublicKeyCredential assertionCredential) throws Exception {
        
        byte[] credentialId = assertionCredential.getId();
        
        AuthenticatorAttestationResponse resp = (AuthenticatorAttestationResponse)assertionCredential.getResponse();
        
        AttestationObject attObj = resp.parseAttestationObject();
        
        AttestationResult ret = this.validateAttestation(resp.getClientData(),attObj);
        
        if (!Arrays.equals(credentialId,attObj.parseAuthData().getAttestedCredentialData().getCredentialId())) {
            throw new SecurityException("Credential IDs do not match in the attestation response for ["+ret+"].");
        }
        
        return ret;
    }

}
