/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 17.06.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.u2f.impl;

import java.security.MessageDigest;
import java.security.Signature;
import java.util.Arrays;

import org.clazzes.login.jbo.common.Helpers;
import org.clazzes.login.jbo.common.PubKeyInfo;
import org.clazzes.login.jbo.u2f.AssertionValidator;
import org.clazzes.login.jbo.u2f.AttestedCredentialData;
import org.clazzes.login.jbo.u2f.AuthenticatorAssertionResponse;
import org.clazzes.login.jbo.u2f.AuthenticatorData;
import org.clazzes.login.jbo.u2f.AuthenticatorDataFactory;
import org.clazzes.login.jbo.u2f.PublicKeyCredential;

/**
 * The implementation of an assertion validator.
 */
public class AssertionValidatorImpl implements AssertionValidator {

    private AuthenticatorDataFactory authenticatorDataFactory;
    

    @Override
    public AuthenticatorData validateAssertion(PublicKeyCredential assertionCredential) throws Exception {
       
        byte[] credentialId = assertionCredential.getId();
        
        AuthenticatorAssertionResponse resp = (AuthenticatorAssertionResponse) assertionCredential.getResponse();
        
        AuthenticatorData assertedData = resp.parseAuthenticatorData();
        
        AuthenticatorData attestedData = this.authenticatorDataFactory.getAuthenticatorData(credentialId);
        
        if (attestedData == null) {
            throw new SecurityException("Cannot find an attestion for credential ID ["+Helpers.formatHex(credentialId)+"].");
        }
        
        if (!Arrays.equals(assertedData.getRpIdHash(),attestedData.getRpIdHash())) {
            throw new SecurityException("Relaying party hashes mismatch for credential ID ["+Helpers.formatHex(credentialId)+"].");
        }
        
        AttestedCredentialData attestedCredentialData = attestedData.getAttestedCredentialData();
        
        PubKeyInfo pubKey = attestedCredentialData.getCredentialPublicKey();
        
        MessageDigest sha256 = MessageDigest.getInstance("SHA-256");
        
        byte[] clientDataHash = sha256.digest(resp.getClientData());
        
        Signature sig = Signature.getInstance(pubKey.getJceAlgorithm());
        
        sig.initVerify(pubKey.getPublicKey());

        sig.update(resp.getAuthenticatorData());
        sig.update(clientDataHash);
        
        boolean result = sig.verify(resp.getSignature());
        
        if (!result) {
            throw new SecurityException("Signatre validation failed for credential ID ["+Helpers.formatHex(credentialId)+"] and public key ["+pubKey+"].");
        }
       
        return attestedData;
    }


    /**
     * @param authenticatorDataFactory The configured authenticator data factory.
     */
    public void setAuthenticatorDataFactory(AuthenticatorDataFactory authenticatorDataFactory) {
        this.authenticatorDataFactory = authenticatorDataFactory;
    }

}
