/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.06.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.util.Arrays;

/**
 * <pre>
 * {
 *         "type": "x509Extension",
 *         "parameters": {
 *           "key": "1.3.6.1.4.1.41482.1.2"
 *         }
 *       }
 * </pre>
 * or
 * <pre>
 *       {
 *         "type": "x509Extension",
 *         "parameters": {
 *           "value": "1.3.6.1.4.1.41482.1.2",
 *           "key": "1.3.6.1.4.1.41482.2"
 *         }
 *       }
 * </pre>
 * or
 * <pre>
 *      {
 *         "type": "x509Extension",
 *         "parameters": {
 *           "key": "1.3.6.1.4.1.45724.1.1.4",
 *           "value": {
 *             "type": "hex",
 *             "value": "c5ef55ffad9a4b9fb580adebafe026d0"
 *           }
 *         }
 *       }
 * </pre>
 */
public class X509DeviceSelector extends DeviceSelector {

    private final String key;
    private final Object value;
    
    public X509DeviceSelector(String key, Object value) {
        super();
        this.key = key;
        this.value = value;
    }

    /**
     * @return The OID of the extension matched.
     */
    public String getKey() {
        return this.key;
    }

    /**
     * @return The extensions value, which might be a string or a byte array.
     */
    public Object getValue() {
        return this.value;
    }

    @Override
    public boolean matches(AttestationCertInfo aci) {
        
        Object v = aci.getExtensionValue(this.key);
        
        if (v instanceof byte[] && this.value instanceof byte[]) {
            return Arrays.equals((byte[])v,(byte[])this.value);
        }
        
        if (this.value == null) {
            return v == null;
        }
        
        return this.value.equals(v);
    }

    @Override
    public String getType() {
        
        return "x509Extension";
    }

}
