/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 11.12.2019
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

/**
 * Token transport flags according to the
 * <a href="https://fidoalliance.org/specs/fido-u2f-v1.2-ps-20170411/fido-u2f-authenticator-transports-extension-v1.2-ps-20170411.html">
 * FIDO U2F Extension Specification</a>.
 */
public interface Transports {

    public static final int TRANSPORT_BLUETOOTH = 1<<0;
    public static final int TRANSPORT_BLUETOOTH_LOW_ENERGY = 1<<1;
    public static final int TRANSPORT_USB = 1<<2;
    public static final int TRANSPORT_NFC = 1<<3;
    public static final int TRANSPORT_USB_INTERNAL = 1<<4;

    /**
     * @param transports A transports bit mask or <code>null</code>.
     * @return the formatted transport bit mask or <code>null</code>, if the input was <code>null</code>.
     */
    public static String format(Integer transports) {
        
        if (transports == null) {
            return null;
        }
        
        return Transports.format(transports.intValue());
    }
    
    /**
     * @param transports A transports bit mask.
     * @return the formatted transport bit mask.
     */
    public static String format(int transports) {
        
        StringBuffer sb = new StringBuffer();
        
        if ((transports & TRANSPORT_BLUETOOTH) != 0) {
            sb.append("Bluetooth");
        }
        
        if ((transports & TRANSPORT_BLUETOOTH_LOW_ENERGY) != 0) {
            if (sb.length() > 0) {
                sb.append("|");
            }
            sb.append("Bluetooth_LE");
        }

        if ((transports & TRANSPORT_NFC) != 0) {
            if (sb.length() > 0) {
                sb.append("|");
            }
            sb.append("NFC");
        }
        
        if ((transports & TRANSPORT_USB) != 0) {
            if (sb.length() > 0) {
                sb.append("|");
            }
            sb.append("USB");
        }

        if ((transports & TRANSPORT_USB_INTERNAL) != 0) {
            if (sb.length() > 0) {
                sb.append("|");
            }
            sb.append("USB_internal");
        }
        
        return sb.toString();
    }

    
    public static boolean isTokenBluetooth(Integer transports) {
        return transports != null && isTokenBluetooth(transports.intValue());
    }

    public static boolean isTokenBluetoothLE(Integer transports) {
        return transports != null && isTokenBluetoothLE(transports.intValue());
    }

    public static boolean isTokenNFC(Integer transports) {
        return transports != null && isTokenNFC(transports.intValue());
    }

    public static boolean isTokenUSB(Integer transports) {
        return transports != null && isTokenUSB(transports.intValue());
    }

    public static boolean isTokenUSBInternal(Integer transports) {
        return transports != null && isTokenUSBInternal(transports.intValue());
    }

    public static boolean isTokenBluetooth(int transports) {
        return (transports & TRANSPORT_BLUETOOTH) != 0;
    }

    public static boolean isTokenBluetoothLE(int transports) {
        return (transports & TRANSPORT_BLUETOOTH_LOW_ENERGY) != 0;
    }

    public static boolean isTokenNFC(int transports) {
        return (transports & TRANSPORT_NFC) != 0;
    }

    public static boolean isTokenUSB(int transports) {
        return (transports & TRANSPORT_USB) != 0;
    }

    public static boolean isTokenUSBInternal(int transports) {
        return (transports & TRANSPORT_USB_INTERNAL) != 0;
    }


}
