/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 09.06.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.util.Arrays;

import org.clazzes.login.jbo.common.Helpers;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * <pre>
 * {
 *   'nonce': '/HmvKLP5yKE/tGhGmCkJNpz7e6RjrYnds/QWjBrQKi8=',
 *   'timestampMs': 1591556402440,
 *   'apkPackageName': 'com.google.android.gms',
 *   'apkDigestSha256': '4vE+NvtX+O73jTnQucjhf0Cd7WIsxWxrvH3GGHwmFHc=',
 *   'ctsProfileMatch': True,
 *   'apkCertificateDigestSha256': ['8P1sW0EPJcslw7UzRsiXL64w+O50Ed+RBICtay1g24M='],
 *   'basicIntegrity': True,
 *   'evaluationType': 'BASIC'
 * }
 * </pre>
 */
public class SafetyNetPayload {

    private final byte[] nonce;
    private final long timestampMs;
    private final String apkPackageName;
    private final byte[] apkDigestSha256;
    private final boolean ctsProfileMatch;
    private final byte[][] apkCertificateDigestSha256;
    private final boolean basicIntegrity;
    private final String evaluationType;

    @JsonCreator(mode = JsonCreator.Mode.PROPERTIES)
    public SafetyNetPayload(
            @JsonProperty("nonce") byte[] nonce,
            @JsonProperty("timestampMs") long timestampMs,
            @JsonProperty("apkPackageName") String apkPackageName,
            @JsonProperty("apkDigestSha256") byte[] apkDigestSha256,
            @JsonProperty("ctsProfileMatch") boolean ctsProfileMatch,
            @JsonProperty("apkCertificateDigestSha256") byte[][] apkCertificateDigestSha256,
            @JsonProperty("basicIntegrity") boolean basicIntegrity,
            @JsonProperty("evaluationType") String evaluationType) {
        super();
        this.nonce = nonce;
        this.timestampMs = timestampMs;
        this.apkPackageName = apkPackageName;
        this.apkDigestSha256 = apkDigestSha256;
        this.ctsProfileMatch = ctsProfileMatch;
        this.apkCertificateDigestSha256 = apkCertificateDigestSha256;
        this.basicIntegrity = basicIntegrity;
        this.evaluationType = evaluationType;
    }

    public byte[] getNonce() {
        return this.nonce;
    }

    public long getTimestampMs() {
        return this.timestampMs;
    }

    public String getApkPackageName() {
        return this.apkPackageName;
    }

    public byte[] getApkDigestSha256() {
        return this.apkDigestSha256;
    }

    public boolean isCtsProfileMatch() {
        return this.ctsProfileMatch;
    }

    public byte[][] getApkCertificateDigestSha256() {
        return this.apkCertificateDigestSha256;
    }

    public boolean isBasicIntegrity() {
        return this.basicIntegrity;
    }

    public String getEvaluationType() {
        return this.evaluationType;
    }

    @Override
    public String toString() {
        return "SafetyNetPayload [nonce=" + Helpers.formatHex(this.nonce) +
                ", timestampMs=" + this.timestampMs +
                ", apkPackageName=" + this.apkPackageName +
                ", apkDigestSha256=" + Helpers.formatHex(this.apkDigestSha256) +
                ", ctsProfileMatch=" + this.ctsProfileMatch +
                ", apkCertificateDigestSha256=" + Arrays.toString(this.apkCertificateDigestSha256) +
                ", basicIntegrity=" + this.basicIntegrity
                + ", evaluationType=" + this.evaluationType + "]";
    }

}
