/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 09.06.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Map;

import org.clazzes.login.jbo.common.Algorithm;
import org.clazzes.login.jbo.common.Helpers;
import org.clazzes.login.jbo.u2f.json.CredentialParser;

/**
 * An attestation statement as issued by Android for fingerprints and the like.
 */
public class SafetyNetAttestationStatement extends AttestationStatement {

    public static final String FORMAT="android-safetynet";

    private final String version;
    private final SafetyNetSignatureHeader header;
    private final SafetyNetPayload payload;
    private final byte[] signaturePayload;
    private final byte[] signature;

    public static SafetyNetAttestationStatement fromCbor(Map<String,Object> attStmt) throws Exception {

        byte[] signature = null;

        String version = (String)attStmt.get("ver");
        SafetyNetSignatureHeader header = null;
        byte[] signaturePayload = null;
        SafetyNetPayload payload = null;

        byte[] jwsResponseRaw = (byte[])attStmt.get("response");

        if (jwsResponseRaw != null) {
            int dpos = jwsResponseRaw.length;

            while (--dpos >= 0) {
                if (jwsResponseRaw[dpos] == '.') {
                    break;
                }
            }

            if (dpos < 0) {
                throw new IllegalArgumentException("android-safetynet attestation statement [response] contains JWS response without a dot.");
            }

            signaturePayload = Arrays.copyOf(jwsResponseRaw,dpos);

            String jwsResponse = new String(jwsResponseRaw,"UTF-8");

            byte[][] jwsTokens = Helpers.splitToken(jwsResponse,3);

            if (jwsTokens.length != 3) {
                throw new IllegalArgumentException("android-safetynet attestation statement [response] contains JWS response with less than 3 parts.");
            }

            header = CredentialParser.parseSafetyNetSignatureHeader(jwsTokens[0]);

            payload = CredentialParser.parseSafetyNetPayload(jwsTokens[1]);
            signature = jwsTokens[2];
        }

        return new SafetyNetAttestationStatement(version,header,payload,signaturePayload,signature);
    }

    public SafetyNetAttestationStatement(String version, SafetyNetSignatureHeader header, SafetyNetPayload payload,
            byte[] signaturePayload, byte[] signature) {
        super();
        this.version = version;
        this.header = header;
        this.payload = payload;
        this.signaturePayload = signaturePayload;
        this.signature = signature;
    }

    @Override
    public String getFormat() {
        return FORMAT;
    }

    @Override
    public byte[] getSignature() {

        return this.signature;
    }

    @Override
    public Algorithm getAlgorithm() {

        return this.header == null ? null : this.header.getAlg();
    }

    @Override
    public X509Certificate[] getCertificateChain() {

        return this.header == null ? null : this.header.getX5c();
    }

    public String getVersion() {
        return this.version;
    }

    public SafetyNetSignatureHeader getHeader() {
        return this.header;
    }

    public SafetyNetPayload getPayload() {
        return this.payload;
    }

    /**
     * @return The signature payload, aka the dot-concatenated base64 encoded,
     *         JSON-serialized header and payload.
     */
    public byte[] getSignaturePayload() {
        return this.signaturePayload;
    }

    @Override
    public String toString() {
        return "SafetyNetAttestationStatement [version=" + this.version +
                ", header=" + this.header +
                ", payload=" + this.payload +
                ", signature=" + Helpers.formatHex(this.signature) + "]";
    }

}
