/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 11.12.2019
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Map;

import org.clazzes.login.jbo.common.Algorithm;
import org.clazzes.login.jbo.common.Helpers;

/**
 * Packed Attestation Statement as of
 * <a href="https://www.w3.org/TR/webauthn/#packed-attestation">
 * § 8.2. of the WebAuthn specification</a>.
 */
public class PackedAttestationStatement extends FidoU2FAttestationStatement {

    public static final String FORMAT = "packed";

    private final Algorithm algorithm;

    public static final PackedAttestationStatement fromCbor(Map<String,Object> attStmt) throws Exception {

        Algorithm algorithm = Algorithm.getByValue((Integer)attStmt.get("alg"));
        byte[] signature = (byte[])attStmt.get("sig");

        X509Certificate[] certificateChain =
            CborHelpers.getCertificateChain(attStmt,"x5c");

        return new PackedAttestationStatement(algorithm,signature,certificateChain);
    }

    public PackedAttestationStatement(Algorithm algorithm, byte[] signature, X509Certificate[] certificateChain) {
        super(signature,certificateChain);
        this.algorithm = algorithm;
    }

    @Override
    public String getFormat() {
        return FORMAT;
    }

    @Override
    public Algorithm getAlgorithm() {
        return this.algorithm;
    }

    @Override
    public String toString() {
        return "PackedAttestationStatement [algorithm=" + this.algorithm +
                ", signature=" + Helpers.formatHex(this.getSignature()) +
                ", certificateChain=" + Arrays.toString(this.getCertificateChain()) + "]";
    }

}
