/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.06.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.security.cert.X509Certificate;
import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * A device registry as it may be downloaded from
 * <a href="https://developers.yubico.com/U2F/yubico-metadata.json">the YubiCo website</a>.
 */
public class DeviceRegistry {

    private final UUID identifier;
    private final Integer version;
    private final VendorInfo vendorInfo;
    private final X509Certificate[] trustedCertificates;
    private final DeviceInfo[] devices;

    @JsonCreator(mode = JsonCreator.Mode.PROPERTIES)
    public DeviceRegistry(
            @JsonProperty("identifier") UUID identifier,
            @JsonProperty("version") Integer version,
            @JsonProperty("vendorInfo") VendorInfo vendorInfo,
            @JsonProperty("trustedCertificates") X509Certificate[] trustedCertificates,
            @JsonProperty("devices") DeviceInfo[] devices) {
        super();
        this.identifier = identifier;
        this.version = version;
        this.vendorInfo = vendorInfo;
        this.trustedCertificates = trustedCertificates;
        this.devices = devices;
    }

    public UUID getIdentifier() {
        return this.identifier;
    }

    public Integer getVersion() {
        return this.version;
    }

    public VendorInfo getVendorInfo() {
        return this.vendorInfo;
    }

    public X509Certificate[] getTrustedCertificates() {
        return this.trustedCertificates;
    }

    public DeviceInfo[] getDevices() {
        return this.devices;
    }

    public DeviceInfo selectDevice(AttestationCertInfo aci) {

        if (this.devices != null) {
            for (DeviceInfo di : this.devices) {
                if (di.matches(aci)) {
                    return di;
                }
            }
        }
        return null;
    }

    public DeviceInfo selectDevice(String id) {

        if (this.devices != null) {
            for (DeviceInfo di : this.devices) {
                if (id.contentEquals(di.getDeviceId())) {
                    return di;
                }
            }
        }
        return null;
    }
}
