/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.06.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * <pre>
 * {
 *   "deviceId": "1.3.6.1.4.1.41482.1.2",
 *   "displayName": "YubiKey NEO/NEO-n",
 *   "transports": 12,
 *   "deviceUrl": "https://support.yubico.com/support/solutions/articles/15000006494-yubikey-neo",
 *   "imageUrl": "https://developers.yubico.com/U2F/Images/NEO.png",
 *   "selectors": [
 *     {
 *       "type": "x509Extension",
 *       "parameters": {
 *         "key": "1.3.6.1.4.1.41482.1.2"
 *       }
 *     },
 *     {
 *       "type": "x509Extension",
 *       "parameters": {
 *         "value": "1.3.6.1.4.1.41482.1.2",
 *         "key": "1.3.6.1.4.1.41482.2"
 *       }
 *     }
 *   ]
 * }
 * </pre>
 */

public class DeviceInfo {

    private final String deviceId;
    private final String displayName;
    private final Integer transports;
    private final String deviceUrl;
    private final String imageUrl;
    private final DeviceSelector[] selectors;

    @JsonCreator(mode = JsonCreator.Mode.PROPERTIES)
    public DeviceInfo(
            @JsonProperty("deviceId") String deviceId,
            @JsonProperty("displayName") String displayName,
            @JsonProperty("transports") Integer transports,
            @JsonProperty("deviceUrl") String deviceUrl,
            @JsonProperty("imageUrl") String imageUrl,
            @JsonProperty("selectors") DeviceSelector[] selectors) {
        super();
        this.deviceId = deviceId;
        this.displayName = displayName;
        this.transports = transports;
        this.deviceUrl = deviceUrl;
        this.imageUrl = imageUrl;
        this.selectors = selectors;
    }

    public String getDeviceId() {
        return this.deviceId;
    }

    public String getDisplayName() {
        return this.displayName;
    }

    public Integer getTransports() {
        return this.transports;
    }

    public String formatTokenTransport() {

        return Transports.format(this.transports);
    }

    public String getDeviceUrl() {
        return this.deviceUrl;
    }

    public String getImageUrl() {
        return this.imageUrl;
    }

    public DeviceSelector[] getSelectors() {
        return this.selectors;
    }

    /**
     * @param aci The attestation certificate info to match.
     * @return Whether all selectors match the given attestation certificate.
     */
    public boolean matches(AttestationCertInfo aci) {

        if (this.selectors == null) {
            return false;
        }

        for (DeviceSelector selector : this.selectors) {
            if (!selector.matches(aci)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public String toString() {
        return "DeviceInfo [devieId=" + this.deviceId +
                ", displayName=" + this.displayName +
                ", transports=" + this.formatTokenTransport() +
                ", deviceUrl=" + this.deviceUrl +
                ", imageUrl=" + this.imageUrl + "]";
    }

}
