/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 11.12.2019
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.io.ByteArrayInputStream;
import java.math.BigInteger;
import java.security.AlgorithmParameters;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.text.ParsePosition;
import java.util.HashMap;
import java.util.Map;

import org.clazzes.login.jbo.bc.BCTools;
import org.clazzes.login.jbo.common.Algorithm;
import org.clazzes.login.jbo.common.CurveType;
import org.clazzes.login.jbo.common.Helpers;
import org.clazzes.login.jbo.common.KeyOperation;
import org.clazzes.login.jbo.common.KeyType;
import org.clazzes.login.jbo.common.PubKeyInfo;
import org.clazzes.login.jbo.cose.COSEKeyTypeParameter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.dataformat.cbor.databind.CBORMapper;

/**
 * A CBOR-encoded public key with a type and key parameters.
 */
public class CredentialPublicKey {

    private static final Logger log = LoggerFactory.getLogger(CredentialPublicKey.class);

    public static final PubKeyInfo parse(byte[] cborData, ParsePosition pos, int len) throws Exception {

        CBORMapper mapper = CBORMapper.builder().build();

        ObjectReader reader = mapper.readerFor(Map.class);

        Map<String,Object> top;

        try (ByteArrayInputStream is = new ByteArrayInputStream(cborData,pos.getIndex(),len)) {

            top = reader.readValue(is);
        }

        if (log.isDebugEnabled()) {
            log.debug("CredentialPublicKey.parse: top = {}",top);
        }

        KeyType keyType = KeyType.getByValue((Integer)top.get("1"));
        String kid = Helpers.formatBase64((byte[])top.get("2"));
        Algorithm algorithm = Algorithm.getByValue((Integer)top.get("3"));
        KeyOperation keyOperations[] = CborHelpers.getKeyOpsArray(top,"4");

        Map<COSEKeyTypeParameter,Object> paramValues = new HashMap<COSEKeyTypeParameter,Object>();

        for (Map.Entry<String,Object> e: top.entrySet()) {

            int n = Integer.parseInt(e.getKey());

            if (n >= 1 && n <= 5) {
                continue;
            }

            COSEKeyTypeParameter parameter = COSEKeyTypeParameter.getByKeyTypeAndLabel(keyType,n);

            paramValues.put(parameter,e.getValue());
        }

        PublicKey pubKey;
        CurveType ct;

        switch (keyType) {
        case RSA: {

            BigInteger n = new BigInteger(1,(byte[])paramValues.get(COSEKeyTypeParameter.RSA_N));
            BigInteger e = new BigInteger(1,(byte[])paramValues.get(COSEKeyTypeParameter.RSA_E));

            RSAPublicKeySpec pks = new RSAPublicKeySpec(n,e);

            KeyFactory kf = KeyFactory.getInstance("RSA");

            ct = null;
            pubKey = kf.generatePublic(pks);
        }
        break;

        case EC2: {

            ct = CurveType.getByValue((Integer)paramValues.get(COSEKeyTypeParameter.EC2_CRV));

            BigInteger x = new BigInteger(1,(byte[])paramValues.get(COSEKeyTypeParameter.EC2_X));

            Object yitem = paramValues.get(COSEKeyTypeParameter.EC2_Y);

            ECPoint p;

            if (yitem instanceof Boolean ytilde) {
                p = BCTools.decompressPoint(ct,x,ytilde);
            }
            else {
                BigInteger y = new BigInteger(1,(byte[])yitem);
                p = new ECPoint(x,y);
            }

            AlgorithmParameters ap = AlgorithmParameters.getInstance("EC");
            ap.init(new ECGenParameterSpec(ct.getJceType()));

            ECParameterSpec ecps = ap.getParameterSpec(ECParameterSpec.class);

            ECPublicKeySpec pks = new ECPublicKeySpec(p,ecps);

            KeyFactory kf = KeyFactory.getInstance("EC");

            pubKey = kf.generatePublic(pks);
        }
        break;

        case OKP:{

            ct = CurveType.getByValue((Integer)paramValues.get(COSEKeyTypeParameter.OKP_CRV));

            if (ct.getJceType() != null) {
                throw new IllegalArgumentException("EC Curve ["+ct.getJceType()+"] specified, where EdEC curve expected.");
            }

            byte[] x = (byte[])paramValues.get(COSEKeyTypeParameter.OKP_X);

            X509EncodedKeySpec pks = new X509EncodedKeySpec(x);

            // for EdEC, the curve type is actually an algorithm.
            KeyFactory kf = KeyFactory.getInstance(ct.getJwkType());

            pubKey = kf.generatePublic(pks);
        }
        break;

        default:
            throw new IllegalArgumentException("Unsupported key type ["+keyType+"] specified.");
        }

        return new PubKeyInfo(kid,keyType,ct,algorithm,keyOperations,pubKey);
    }

}
