/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 11.12.2019
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.text.ParsePosition;
import java.util.Arrays;

import org.clazzes.login.jbo.common.Helpers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Authenticator data according to
 * <a href="https://www.w3.org/TR/webauthn/#authenticator-data">
 * § 6.1. of the WebAuthn specification</a>.
 */
public class AuthenticatorData {

    private static final Logger log = LoggerFactory.getLogger(AuthenticatorData.class);

    /**
     * User Present flag, bit 0.
     */
    public static final int FLAG_UP = 1 << 0;
    /**
     * User Verified flag, bit 2.
     */
    public static final int FLAG_UV = 1 << 2;
    /**
     * Attested credential data included flag, bit 6.
     */
    public static final int FLAG_AT = 1 << 6;
    /**
     * Extension data included flag, bit 7.
     */
    public static final int FLAG_ED = 1 << 7;


    private final byte[] rpIdHash;
    private final int flags;
    private final long signCount;
    private final AttestedCredentialData attestedCredentialData;

    public static AuthenticatorData parse(byte[] in) throws Exception {

        ParsePosition pos = new ParsePosition(0);

        if (pos.getIndex() + 32 > in.length) {
            throw new IllegalArgumentException("Authenticator Data of length ["+in.length+"] is too short for 32-byte rpIdHash.");
        }

        byte[] rpIdHash = Arrays.copyOfRange(in,pos.getIndex(),pos.getIndex()+32);

        pos.setIndex(pos.getIndex()+32);

        if (pos.getIndex() + 1 > in.length) {
            throw new IllegalArgumentException("Authenticator Data of length ["+in.length+"] is too short for 1-byte flags.");
        }

        int flags = in[pos.getIndex()] & 0xff;

        pos.setIndex(pos.getIndex()+1);

        if (pos.getIndex() + 4 > in.length) {
            throw new IllegalArgumentException("Authenticator Data of length ["+in.length+"] is too short for 4-byte signCount.");
        }

        long signCount =
                ((in[pos.getIndex()] & 0xffL) << 24) |
                ((in[pos.getIndex()+1] & 0xffL) << 16) |
                ((in[pos.getIndex()+2] & 0xffL) << 8) |
                (in[pos.getIndex()+3] & 0xffL);

        pos.setIndex(pos.getIndex()+4);

        AttestedCredentialData attestedCredentialData = null;

        if (pos.getIndex() < in.length) {
            attestedCredentialData = AttestedCredentialData.parse(in, pos,in.length - pos.getIndex());
        }

        if (pos.getIndex() < in.length) {
            log.warn("Ignoring [{}] bytes of extension data",in.length-pos.getIndex());
        }

        return new AuthenticatorData(rpIdHash,flags,signCount,attestedCredentialData);
    }

    public AuthenticatorData(byte[] rpIdHash, int flags, long signCount,
            AttestedCredentialData attestedCredentialData) {
        super();
        this.rpIdHash = rpIdHash;
        this.flags = flags;
        this.signCount = signCount;
        this.attestedCredentialData = attestedCredentialData;
    }

    public byte[] getRpIdHash() {
        return this.rpIdHash;
    }

    public int getFlags() {
        return this.flags;
    }

    public String formatFlags() {

        StringBuffer sb = new StringBuffer();

        if (this.isUserPresent()) {
            sb.append("UP");
        }

        if (this.isUserVerified()) {
            if (sb.length() > 0) {
                sb.append("|");
            }
            sb.append("UV");
        }

        if (this.isAttestedCredentialDataPresent()) {
            if (sb.length() > 0) {
                sb.append("|");
            }
            sb.append("AT");
        }

        if (this.isExtensionDataPresent()) {
            if (sb.length() > 0) {
                sb.append("|");
            }
            sb.append("ED");
        }

        return sb.toString();
    }

    public boolean isUserPresent() {
        return (this.flags & FLAG_UP) != 0;
    }

    public boolean isUserVerified() {
        return (this.flags & FLAG_UV) != 0;
    }

    public boolean isAttestedCredentialDataPresent() {
        return (this.flags & FLAG_AT) != 0;
    }

    public boolean isExtensionDataPresent() {
        return (this.flags & FLAG_ED) != 0;
    }

    public long getSignCount() {
        return this.signCount;
    }

    public AttestedCredentialData getAttestedCredentialData() {
        return this.attestedCredentialData;
    }

    @Override
    public String toString() {
        return "AuthenticatorData [rpIdHash=" + Helpers.formatHex(this.rpIdHash) +
                ", flags=" + this.formatFlags() +
                ", signCount=" + this.signCount +
                ", attestedCredentialData=" + this.attestedCredentialData + "]";
    }



}
