/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 11.12.2019
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.text.ParsePosition;
import java.util.Arrays;

import org.clazzes.login.jbo.common.Helpers;
import org.clazzes.login.jbo.common.PubKeyInfo;

/**
 * Attested Credential Data according to
 * <a href="https://www.w3.org/TR/webauthn/#sec-attested-credential-data">
 * § 6.4.1. of the WebAuthn specification</a>.
 */
public class AttestedCredentialData {

    private final byte[] aaguid;
    private final byte[] credentialId;
    private final PubKeyInfo credentialPublicKey;

    public static final AttestedCredentialData parse(byte[] in, ParsePosition pos, int len) throws Exception {

        int end = pos.getIndex() + len;

        if (pos.getIndex() + 16 > end) {
            throw new IllegalArgumentException("Attested Credential Data of length ["+len+"] is too short for 16-byte aaguid.");
        }

        byte[] aaguid = Arrays.copyOfRange(in,pos.getIndex(),pos.getIndex()+16);

        pos.setIndex(pos.getIndex()+16);

        if (pos.getIndex() + 2 > end) {
            throw new IllegalArgumentException("Attested Credential Data of length ["+len+"] is too short for 2-byte credentialId length.");
        }

        int l = ((in[pos.getIndex()] & 0xff) << 8) + (in[pos.getIndex()+1] & 0xff);

        pos.setIndex(pos.getIndex()+2);

        if (pos.getIndex() + l > end) {
            throw new IllegalArgumentException("Attested Credential Data of length ["+len+"] is too short for credentialId of length ["+l+"].");
        }

        byte[] credentialId = Arrays.copyOfRange(in,pos.getIndex(),pos.getIndex()+l);

        pos.setIndex(pos.getIndex()+l);

        PubKeyInfo credentialPublicKey = CredentialPublicKey.parse(in,pos,end-pos.getIndex());

        return new AttestedCredentialData(aaguid,credentialId,credentialPublicKey);
    }

    public AttestedCredentialData(byte[] aaguid, byte[] credentialId, PubKeyInfo credentialPublicKey) {
        super();
        this.aaguid = aaguid;
        this.credentialId = credentialId;
        this.credentialPublicKey = credentialPublicKey;
    }

    /**
     * @return The public key for these credentials.
     */
    public PubKeyInfo getCredentialPublicKey() {
        return credentialPublicKey;
    }

    /**
     * @return The 16-byte AAGUID of the authenticator.
     */
    public byte[] getAaguid() {
        return aaguid;
    }

    /**
     * @return The credential ID.
     */
    public byte[] getCredentialId() {
        return credentialId;
    }

    @Override
    public String toString() {
        return "AttestedCredentialData [aaguid=" + Helpers.formatHex(this.aaguid) +
                ", credentialId=" + Helpers.formatHex(this.credentialId) +
                ", credentialPublicKey=" + this.credentialPublicKey + "]";
    }


}
