/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 11.12.2019
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.util.Map;

import org.clazzes.login.jbo.common.Helpers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.dataformat.cbor.databind.CBORMapper;

/**
 * An attestation object, see
 * <a href="https://www.w3.org/TR/webauthn/#attestation-object">the W3C WebAuthn specification</a>
 * for details.
 */
public class AttestationObject {

    private static final Logger log = LoggerFactory.getLogger(AttestationObject.class);

    private final byte[] authData;
    private final AttestationStatement attStmt;

    private AuthenticatorData parsedAuthData;

    public static AttestationObject parse(byte[] cborData) throws Exception {

        CBORMapper mapper = CBORMapper.builder().build();

        ObjectReader reader = mapper.readerFor(Map.class);

        Map<String,Object> top = reader.readValue(cborData);

        if (log.isDebugEnabled()) {
            // top = {fmt=fido-u2f, attStmt={sig=[B@28f3b248, x5c=[[B@1b1426f4]}, authData=[B@32b260fa}
            log.debug("AttestationObject.parse: top = {}",top);
        }

        byte[] authData = (byte[])top.get("authData");
        @SuppressWarnings("unchecked")
        Map<String,Object> attStmtItem = (Map<String,Object>)top.get("attStmt");
        String attStmtFmt = (String)top.get("fmt");
        AttestationStatement attStmt = null;

        if (PackedAttestationStatement.FORMAT.equals(attStmtFmt)) {

            attStmt = PackedAttestationStatement.fromCbor(attStmtItem);
        }
        else if (FidoU2FAttestationStatement.FORMAT.equals(attStmtFmt)) {

            attStmt = FidoU2FAttestationStatement.fromCbor(attStmtItem);
        }
        else if (SafetyNetAttestationStatement.FORMAT.equals(attStmtFmt)) {

            attStmt = SafetyNetAttestationStatement.fromCbor(attStmtItem);
        }
        else {
            throw new IllegalArgumentException("Unsupported attestation statement format ["+attStmtFmt+"] in attestation object.");
        }

        return new AttestationObject(authData,attStmt);
    }

    public AttestationObject(byte[] authData, AttestationStatement attStmt) {
        super();
        this.authData = authData;
        this.attStmt = attStmt;
    }

    public byte[] getAuthData() {
        return this.authData;
    }

    public AuthenticatorData parseAuthData() throws Exception {

        if (this.authData == null) {
            return null;
        }

        if (this.parsedAuthData == null) {
            this.parsedAuthData = AuthenticatorData.parse(this.authData);
        }

        return this.parsedAuthData;
    }

    public AttestationStatement getAttStmt() {
        return this.attStmt;
    }

    @Override
    public String toString() {
        return "AttestationObject [authData=" + Helpers.formatHex(this.authData) + ", attStmt=" + this.attStmt + "]";
    }

}
