/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 26.05.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.u2f;

import java.util.Map;

import org.clazzes.login.jbo.common.Helpers;
import org.clazzes.login.jbo.common.PubKeyInfo;

/**
 * Parsed information of an attestation certificate.
 */
public class AttestationCertInfo {

    public static final String idFidoGenCeAaguid = "1.3.6.1.4.1.45724.1.1.4";
    public static final String idFidoU2FTransports = "1.3.6.1.4.1.45724.2.1.1";
    public static final String idYubicoTokenType = "1.3.6.1.4.1.41482.2";
        
    private final Map<String,Object> extensionValues;
    private final PubKeyInfo pubKeyInfo;
        
    public AttestationCertInfo(Map<String,Object> extensionValues, PubKeyInfo pubKeyInfo) {
        super();
        this.extensionValues = extensionValues;
        this.pubKeyInfo = pubKeyInfo;
    }

    public String getTokenType() {
        return (String)this.extensionValues.get(idYubicoTokenType);
    }

    public byte[] getAaguid() {
        return (byte[])this.extensionValues.get(idFidoGenCeAaguid);
    }

    /**
     * @return The token transport with flags according to the
     *         <a href="https://fidoalliance.org/specs/fido-u2f-v1.2-ps-20170411/fido-u2f-authenticator-transports-extension-v1.2-ps-20170411.html">
     *         FIDO U2F Extension Specification</a>.
     * @see Transports
     */
    public Integer getTokenTransport() {
        return (Integer)this.extensionValues.get(idFidoU2FTransports);
    }

    public String formatTokenTransport() {
        
        return Transports.format(this.getTokenTransport());
    }
    
    public PubKeyInfo getPubKeyInfo() {
        return this.pubKeyInfo;
    }
    
    public Object getExtensionValue(String oid) {
        return this.extensionValues.get(oid);
    }

    @Override
    public String toString() {
        
        byte[] aaguid = this.getAaguid();
        
        return "AttestationCertInfo [tokenType=" + this.getTokenType() +
                " ,tokenTransport=" + this.formatTokenTransport() +
                " ,aaguid=" + (aaguid == null ? null : Helpers.formatHex(aaguid)) +
                ", pubKeyInfo=" + this.pubKeyInfo + "]";
    }
    
}
