/***********************************************************
 * $Id$
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 28.05.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.jwt;

import java.util.List;
import java.util.Map;

/**
 * <p>
 * The claims of an <a href="https://tools.ietf.org/html/rfc7519">RFC 7519</a> compatible
 * JSON Web Token.
 * </p>
 *
 * <pre>
 * {"aud":"6731de76-14a6-49ae-97bc-6eba6914391e",
 *  "iss":"https://login.microsoftonline.com/b9419818-09af-49c2-b0c3-653adc1f376e/v2.0",
 *  "iat":1452285331,
 *  "nbf":1452285331,
 *  "exp":1452289231,
 *  "name":"Babe Ruth",
 *  "nonce":"12345",
 *  "oid":"a1dbdde8-e4f9-4571-ad93-3059e3750d23",
 *  "preferred_username":"thegreatbambino@nyy.onmicrosoft.com",
 *  "sub":"MF4f-ggWMEji12KynJUNQZphaUTvLcQug5jdF2nl01Q",
 *  "tid":"b9419818-09af-49c2-b0c3-653adc1f376e",
 *  "ver":"2.0"}
 * </pre>
 */
public class JWTokenClaims {

    /*
     * iss     Issuer  [IESG]  [ RFC7519, Section 4.1.1]
     * sub     Subject [IESG]  [ RFC7519, Section 4.1.2]
     * aud     Audience        [IESG]  [ RFC7519, Section 4.1.3]
     * exp     Expiration Time [IESG]  [ RFC7519, Section 4.1.4]
     * nbf     Not Before      [IESG]  [ RFC7519, Section 4.1.5]
     * iat     Issued At       [IESG]  [ RFC7519, Section 4.1.6]
     * jti     JWT ID
     */
    private final String jwtId;
    private final String issuer;
    private final String subject;
    private final List<String> audience;
    private final Long issuedAt;
    private final Long notBefore;
    private final Long expiration;

    private final Map<String,Object> additionalClaims;

    public JWTokenClaims(String jwtId, String issuer, String subject,
            List<String> audience, Long issuedAt, Long notBefore, Long expirationTime,
            Map<String,Object> additionalClaims) {
        super();
        this.jwtId = jwtId;
        this.issuer = issuer;
        this.subject = subject;
        this.audience = audience;
        this.issuedAt = issuedAt;
        this.notBefore = notBefore;
        this.expiration = expirationTime;
        this.additionalClaims = additionalClaims;
    }

    /**
     * @return The JWT token ID as parsed from the <code>jti</code> property.
     */
    public String getJwtId() {
        return this.jwtId;
    }

    /**
     * @return The issuer as parsed from the <code>iss</code> property.
     */
    public String getIssuer() {
        return this.issuer;
    }

    /**
     * @return The subject as parsed from the <code>sub</code> property.
     */
    public String getSubject() {
        return this.subject;
    }

    /**
     * @return The audience as parsed from the <code>aud</code> property.
     */
    public List<String> getAudience() {
        return this.audience;
    }

    /**
     * @return The issuing timestamp as parsed from the <code>iat</code> property
     *          in milliseconds since the epoch.
     */
    public Long getIssuedAt() {
        return this.issuedAt;
    }

    /**
     * @return The timestamp of the start of validity as parsed from the <code>nbf</code>
     *         property in milliseconds since the epoch.
     */
    public Long getNotBefore() {
        return this.notBefore;
    }

    /**
     * @return The expiration timestamp as parsed from the <code>exp</code> property
     *         in milliseconds since the epoch.
     */
    public Long getExpiration() {
        return this.expiration;
    }

    /**
     * @return The map of additional claims representing all additional properties of the JSON object.
     */
    public Map<String,Object> getAdditionalClaims() {
        return this.additionalClaims;
    }

    /**
     * Return the additional claim for the given key.
     * @param key The key of the additional, vendor-specific claims.
     * @return The value of the given claim or <code>null</code>, if no such
     *         claim was contained in the Token.
     */
    public Object getAdditionalClaim(String key) {
        return this.additionalClaims.get(key);
    }

    @Override
    public String toString() {
        return "JWTokenClaims [jwtId=" + this.jwtId + ", issuer=" + this.issuer
                + ", subject=" + this.subject + ", audience=" + this.audience
                + ", issuedAt=" + this.issuedAt + ", notBefore=" + this.notBefore
                + ", expiration=" + this.expiration + ", additionalClaims="
                + this.additionalClaims + "]";
    }
}
