/***********************************************************
 * $Id$
 * 
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 28.05.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.jwt;

import org.clazzes.login.jbo.common.Algorithm;

/**
 * <p>
 * An <a href="https://tools.ietf.org/html/rfc7519">RFC 7519</a> compatible
 * JSON Web Token.
 * </p>
 * 
 * <pre>
 * echo "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiIsImtpZCI6Ik1uQ19WWmNBVGZNNXBPWWlKSE1iYTlnb0VLWSJ9.eyJhdWQiOiI2NzMxZGU3Ni0xNGE2LTQ5YWUtOTdiYy02ZWJhNjkxNDM5MWUiLCJpc3MiOiJodHRwczovL2xvZ2luLm1pY3Jvc29mdG9ubGluZS5jb20vYjk0MTk4MTgtMDlhZi00OWMyLWIwYzMtNjUzYWRjMWYzNzZlL3YyLjAiLCJpYXQiOjE0NTIyODUzMzEsIm5iZiI6MTQ1MjI4NTMzMSwiZXhwIjoxNDUyMjg5MjMxLCJuYW1lIjoiQmFiZSBSdXRoIiwibm9uY2UiOiIxMjM0NSIsIm9pZCI6ImExZGJkZGU4LWU0ZjktNDU3MS1hZDkzLTMwNTllMzc1MGQyMyIsInByZWZlcnJlZF91c2VybmFtZSI6InRoZWdyZWF0YmFtYmlub0BueXkub25taWNyb3NvZnQuY29tIiwic3ViIjoiTUY0Zi1nZ1dNRWppMTJLeW5KVU5RWnBoYVVUdkxjUXVnNWpkRjJubDAxUSIsInRpZCI6ImI5NDE5ODE4LTA5YWYtNDljMi1iMGMzLTY1M2FkYzFmMzc2ZSIsInZlciI6IjIuMCJ9.p_rYdrtJ1oCmgDBggNHB9O38KTnLCMGbMDODdirdmZbmJcTHiZDdtTc-hguu3krhbtOsoYM2HJeZM3Wsbp_YcfSKDY--X_NobMNsxbT7bqZHxDnA2jTMyrmt5v2EKUnEeVtSiJXyO3JWUq9R0dO-m4o9_8jGP6zHtR62zLaotTBYHmgeKpZgTFB9WtUq8DVdyMn_HSvQEfz-LWqckbcTwM_9RNKoGRVk38KChVJo4z5LkksYRarDo8QgQ7xEKmYmPvRr_I7gvM2bmlZQds2OeqWLB1NSNbFZqyFOCgYn3bAQ-nEQSKwBaA36jYGPOVG2r2Qv1uKcpSOxzxaQybzYpQ" | tr -- "-_." "+/\n"
 * 
 * {"typ":"JWT","alg":"RS256","kid":"MnC_VZcATfM5pOYiJHMba9goEKY"}
 * 
 * {"aud":"6731de76-14a6-49ae-97bc-6eba6914391e",
 *  "iss":"https://login.microsoftonline.com/b9419818-09af-49c2-b0c3-653adc1f376e/v2.0",
 *  "iat":1452285331,
 *  "nbf":1452285331,
 *  "exp":1452289231,
 *  "name":"Babe Ruth",
 *  "nonce":"12345",
 *  "oid":"a1dbdde8-e4f9-4571-ad93-3059e3750d23",
 *  "preferred_username":"thegreatbambino@nyy.onmicrosoft.com",
 *  "sub":"MF4f-ggWMEji12KynJUNQZphaUTvLcQug5jdF2nl01Q",
 *  "tid":"b9419818-09af-49c2-b0c3-653adc1f376e",
 *  "ver":"2.0"}
 * </pre>
 */
public class JWToken {

    private final Algorithm algorithm;
    private final String keyId;
    private final JWTokenClaims claimSet;
    private final byte[] signaturePayload;
    private final byte[] signature;
    
    public JWToken(Algorithm algorithm, String keyId, JWTokenClaims claimSet,
            byte[] signaturePayload,
            byte[] signature) {
        super();
        this.algorithm = algorithm;
        this.keyId = keyId;
        this.claimSet = claimSet;
        this.signaturePayload = signaturePayload;
        this.signature = signature;
    }

    /**
     * @return The JCE algorithm name, which is the translated name of the
     *         <code>alg</code> JSON property, e.g. <code>SHA256withRSA</code> (JCE)
     *         for <code>RS256</code> (JWK).
     */
    public Algorithm getAlgorithm() {
        return this.algorithm;
    }

    /**
     * @return The key ID used to sign the token as given in the <code>kid</code>
     *         JSON property.
     */
    public String getKeyId() {
        return this.keyId;
    }

    /**
     * @return The representation of the set of claims for the token.
     */
    public JWTokenClaims getClaimSet() {
        return this.claimSet;
    }

    /**
     * @return The signature payload composed of the base64-encoded header
     *         plus a dot plus the base64-encoded claim set.
     */
    public byte[] getSignaturePayload() {
        return this.signaturePayload;
    }

    /**
     * @return The signature on the claim set generated with the specified algorithm
     *         and key ID.
     */
    public byte[] getSignature() {
        return this.signature;
    }

}
