/***********************************************************
 * $Id$
 * 
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 28.05.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.jwt;

import java.security.cert.Certificate;
import java.util.Arrays;

import org.clazzes.login.jbo.common.PubKeyInfo;

/**
 * <p>
 * An <a href="https://tools.ietf.org/html/rfc7517">RFC 7515</a> compatible
 * public JSON Web Key.
 * </p>
 *
 * <pre>
 * {"keys":
       [
         {"kty":"EC",
          "crv":"P-256",
          "x":"MKBCTNIcKUSDii11ySs3526iDZ8AiTo7Tu6KPAqv7D4",
          "y":"4Etl6SRW2YiLUrN5vfvVHuhp7x8PxltmWWlbbM4IFyM",
          "use":"enc",
          "kid":"1"},

         {"kty":"RSA",
          "n": "0vx7agoebGcQSuuPiLJXZptN9nndrQmbXEps2aiAFbWhM78LhWx
     4cbbfAAtVT86zwu1RK7aPFFxuhDR1L6tSoc_BJECPebWKRXjBZCiFV4n3oknjhMs
     tn64tZ_2W-5JsGY4Hc5n9yBXArwl93lqt7_RN5w6Cf0h4QyQ5v-65YGjQR0_FDW2
     QvzqY368QQMicAtaSqzs8KJZgnYb9c7d0zgdAZHzu6qMQvRL5hajrn1n91CbOpbI
     SD08qNLyrdkt-bFTWhAI4vMQFh6WeZu0fM4lFd2NcRwr3XPksINHaQ-G_xBniIqb
     w0Ls1jF44-csFCur-kEgU8awapJzKnqDKgw",
          "e":"AQAB",
          "alg":"RS256",
          "kid":"2011-04-29"}
       ]
     }
 * </pre>
 */
public class JWKPubKey {

    private final PubKeyInfo pubKeyInfo;
    
    private final String certificateUrl;
    private final Certificate certificateChain[];
    private final byte[] certificateThumbprint;
    private final byte[] certificateThumbprintSha256;
    
    public JWKPubKey(PubKeyInfo pubKeyInfo, String certificateUrl,
            Certificate[] certificateChain, byte[] certificateThumbprint,
            byte[] certificateThumbprintSha256) {
        super();
        this.pubKeyInfo = pubKeyInfo;
        this.certificateUrl = certificateUrl;
        this.certificateChain = certificateChain;
        this.certificateThumbprint = certificateThumbprint;
        this.certificateThumbprintSha256 = certificateThumbprintSha256;
    }

    /**
     * @return The public key info.
     */
    public PubKeyInfo getPubKeyInfo() {
        return this.pubKeyInfo;
    }

    /**
     * @return The X.509 certificate URL as of the <code>x5u</code> JSON property.
     */
    public String getCertificateUrl() {
        return this.certificateUrl;
    }

    /**
     * @return The X.509 certificate chain as parsed from <code>x5c</code> JSON property.
     *         The first certificate in the chain corresponds to the public key of this
     *         certificate. 
     */
    public Certificate[] getCertificateChain() {
        return this.certificateChain;
    }

    /**
     * @return The SHA-1 hash of the DER-encoded certificate corresponding to the public key,
     *         which might be the first certificate contained in the parsed certificate chain
     *         or the first certificate contained returned by the certificate URL.
     */
    public byte[] getCertificateThumbprint() {
        return this.certificateThumbprint;
    }
    
    /**
     * @return The SHA-1 hash of the DER-encoded certificate corresponding to the public key,
     *         which might be the first certificate contained in the parsed certificate chain
     *         or the first certificate contained returned by the certificate URL.
     */
    public byte[] getCertificateThumbprintSha256() {
        return this.certificateThumbprintSha256;
    }

    @Override
    public String toString() {
        return "JWKPubKey [pubKeyInfo=" + pubKeyInfo + ", certificateUrl=" + certificateUrl + ", certificateChain="
                + Arrays.toString(certificateChain) + "]";
    }
    
}
