/***********************************************************
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 29.03.2025
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.json;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;

public abstract class JsonHelper {

    /**
     * Expect an object start in the JSON stream.
     * @param parser The JSON parser
     * @param ctxt The context to include in error messages.
     * @throws IOException
     */
    public static void beginObject(JsonParser parser, String ctxt) throws IOException {

        if (parser.currentToken() == JsonToken.START_OBJECT) {
            return;
        }

        if (parser.nextToken() != JsonToken.START_OBJECT) {
            throw new IllegalStateException("Expected JSON BEGIN_OBJECT for ["+ctxt+"/"+parser.currentTokenLocation()+"]");
        }
    }

    /**
     * Expect an object end in the JSON stream.
     * @param parser The JSON parser
     * @param ctxt The context to include in error messages.
     * @throws IOException
     */
    public static void endObject(JsonParser parser, String ctxt) throws IOException {

        if (parser.currentToken() != JsonToken.END_OBJECT) {
            throw new IllegalStateException("Expected JSON END_OBJECT for ["+ctxt+"/"+parser.currentTokenLocation()+"]");
        }
    }
    /**
     * Expect an object start or null in the JSON stream.
     * @param parser The JSON parser
     * @param ctxt The context to include in error messages.
     * @return Wether a object start has been ecountered.
     * @throws IOException
     */
    public static boolean beginObjectOrNull(JsonParser parser, String ctxt) throws IOException {

        JsonToken token = parser.nextToken();

        if (token == JsonToken.START_OBJECT) {
            return true;
        }

        if (token != JsonToken.VALUE_NULL) {
            throw new IllegalStateException("Expected JSON BEGIN_OBJECT or VALUE_NULL for ["+ctxt+"/"+parser.currentTokenLocation()+"]");
        }

        return false;
    }

    /**
     * Expect an array start in the JSON stream.
     * @param parser The JSON parser
     * @param ctxt The context to include in error messages.
     * @throws IOException
     */
    public static void beginArray(JsonParser parser, String ctxt) throws IOException {

        if (parser.nextToken() != JsonToken.START_ARRAY) {
            throw new IllegalStateException("Expected JSON BEGIN_ARRAY for ["+ctxt+"/"+parser.currentTokenLocation()+"]");
        }
    }

    /**
     * Expect an array end in the JSON stream.
     * @param parser The JSON parser
     * @param ctxt The context to include in error messages.
     * @throws IOException
     */
    public static void endArray(JsonParser parser, String ctxt) throws IOException {

        if (parser.currentToken() != JsonToken.END_ARRAY) {
            throw new IllegalStateException("Expected JSON END_ARRAY for ["+ctxt+"/"+parser.currentTokenLocation()+"]");
        }
    }

    /**
     * Expect an object end or a field name in the JSON stream.
     * @param parser The JSON parser.
     * @param ctxt The context to include in error messages.
     * @return The next field name or null, if we encountered an
     *         END_OBJECT token.
     * @throws IOException
     */
    public static String nextName(JsonParser parser, String ctxt) throws IOException {

        JsonToken token = parser.nextToken();

        if (token == JsonToken.FIELD_NAME) {
            return parser.currentName();
        }

        if (token != JsonToken.END_OBJECT) {
            throw new IllegalStateException("Expected JSON FIELD_NAME or END_OBJECT for ["+ctxt+"/"+parser.currentTokenLocation()+"]");
        }

        return null;
    }

    /**
     * Expect a string constant in the JSON stream.
     * @param parser The JSON parser.
     * @param ctxt The context to include in error messages.
     * @return The string value.
     * @throws IOException
     */
    public static String nextString(JsonParser parser, String ctxt) throws IOException {

        if (parser.nextToken() != JsonToken.VALUE_STRING) {

            throw new IllegalStateException("Expected VALUE_STRING for ["+ctxt+"/"+parser.currentTokenLocation()+"]");
        }

        return parser.getText();
    }

    /**
     * Expect an integer constant in the JSON stream.
     * @param parser The JSON parser.
     * @param ctxt The context to include in error messages.
     * @return The long integer value.
     * @throws IOException
     */
    public static long nextLong(JsonParser parser, String ctxt) throws IOException {

        if (parser.nextToken() != JsonToken.VALUE_NUMBER_INT) {

            throw new IllegalStateException("Expected VALUE_NUMBER_INT for ["+ctxt+"/"+parser.currentTokenLocation()+"]");
        }

        return parser.getLongValue();
    }
}
