/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 08.12.2019
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.cose;

import java.util.HashMap;
import java.util.Map;

import org.clazzes.login.jbo.common.KeyType;

/**
 * Key type parameters, see
 * <a href="https://www.iana.org/assignments/cose/cose.xhtml#key-type-parameters">The COSE specification</a>
 * for details.
 */
public enum COSEKeyTypeParameter {
    
    /**
     * EC identifier - Taken from the "COSE Elliptic Curves" registry  [RFC8152]
     */
    OKP_CRV(KeyType.OKP,-1),
    /**
     * x-coordinate    [RFC8152]
     */
    OKP_X(KeyType.OKP,-2),
    /**
     * Private key     [RFC8152]
     */
    OKP_D(KeyType.OKP,-4),
    /**
     * EC identifier - Taken from the "COSE Elliptic Curves" registry  [RFC8152][RFC Errata 5650]
     */
    EC2_CRV(KeyType.EC2,-1),
    /**
     * x-coordinate    [RFC8152]
     */
    EC2_X(KeyType.EC2,-2),
    /**
     * y-coordinate    [RFC8152]
     */
    EC2_Y(KeyType.EC2,-3),
    /**
     * Private key     [RFC8152]
     */
    EC2_D(KeyType.EC2,-4),
    /**
     * the RSA modulus n       [RFC8230]
     */
    RSA_N(KeyType.RSA,-1),
    /**
     * the RSA public exponent e       [RFC8230]
     */
    RSA_E(KeyType.RSA,-2),
    /**
     * the RSA private exponent d      [RFC8230]
     */
    RSA_D(KeyType.RSA,-3),
    /**
     * the prime factor p of n [RFC8230]
     */
    RSA_P(KeyType.RSA,-4),
    /**
     * the prime factor q of n [RFC8230]
     */
    RSA_Q(KeyType.RSA,-5),
    /**
     * dP is d mod (p - 1)     [RFC8230]
     */
    RSA_DP(KeyType.RSA,-6),
    /**
     * dQ is d mod (q - 1)     [RFC8230]
     */
    RSA_DQ(KeyType.RSA,-7),
    /**
     * qInv is the CRT coefficient q^(-1) mod p        [RFC8230]
     */
    RSA_QINV(KeyType.RSA,-8),
    /**
     * other prime infos, an array     [RFC8230]
     */
    RSA_OTHER(KeyType.RSA,-9),
    /**
     * a prime factor r_i of n, where i >= 3   [RFC8230]
     */
    RSA_R_I(KeyType.RSA,-10),
    /**
     * d_i = d mod (r_i - 1)   [RFC8230]
     */
    RSA_D_I(KeyType.RSA,-11),
    /**
     * the CRT coefficient t_i = (r_1 * r_2 * ... * r_(i-1))^(-1) mod r_i      [RFC8230]
     */
    RSA_T_I(KeyType.RSA,-12),
    /**
     * Key Value       [RFC8152]
     */
    Symmetric_K(KeyType.Symmetric,-1);
    
    private static final Map<Integer,COSEKeyTypeParameter> paramsByKeyTypeAndLabel;
    
    static {
        
        paramsByKeyTypeAndLabel = new HashMap<Integer,COSEKeyTypeParameter>();
        
        for (COSEKeyTypeParameter p : COSEKeyTypeParameter.values()) {
            
            paramsByKeyTypeAndLabel.put(p.getKeyType().getValue()*256+p.getLabel(),p);
        }
    }
    
    public static final COSEKeyTypeParameter getByKeyTypeAndLabel(KeyType keyType, int label) {
        
        if (keyType == null) {
            throw new NullPointerException("keyType must not be null.");
        }
        
        if (label > -1 || label < -255) {
            throw new IllegalArgumentException("key type paramter label must be a negative number in the range [-255,-1].");
        }
        
        COSEKeyTypeParameter p = paramsByKeyTypeAndLabel.get(keyType.getValue()*256+label);
        
        if (p == null) {
            throw new IllegalArgumentException("The label ["+label+"] is not a known parameter for COSE key type ["+keyType+"].");
        }
        
        return p;
    }
    
    private final KeyType keyType;
    private final int label;
    
    private COSEKeyTypeParameter(KeyType keyType, int label) {
        this.keyType = keyType;
        this.label = label;
    }

    /**
     * @return The key tpy to which this parameter applies.
     */
    public KeyType getKeyType() {
        return this.keyType;
    }

    /**
     * @return The label of this key parameter.
     */
    public int getLabel() {
        return this.label;
    }
    
    
}
