/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 20.05.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.common;

import java.security.PublicKey;
import java.security.Signature;
import java.util.Arrays;

/**
 * Public Key info as parsed from a U2F CredentialPublicKey or parsed
 * as part of a JWK key.
 */
public class PubKeyInfo {

    private final String keyId;
    private final KeyType keyType;
    private final CurveType curve;
    private final Algorithm algorithm;
    private final KeyOperation keyOperations[];
    private final PublicKey publicKey;
    
    public PubKeyInfo(String keyId, KeyType keyType, CurveType curve, Algorithm algorithm, KeyOperation[] keyOperations,
            PublicKey publicKey) {
        super();
        this.keyId = keyId;
        this.keyType = keyType;
        this.curve = curve;
        this.algorithm = algorithm;
        this.keyOperations = keyOperations;
        this.publicKey = publicKey;
    }

    /**
     * @return The key ID, which might be a base64-formatted binary ID.
     */
    public String getKeyId() {
        return this.keyId;
    }

    /**
     * @return The key type.
     */
    public KeyType getKeyType() {
        return this.keyType;
    }

    /**
     * @return The elliptic curve type, if the key type if {@link KeyType#EC2} or
     *         <code>null</code> otherwise.
     */
    public CurveType getCurve() {
        return this.curve;
    }

    /**
     * @return The algorithm used for signing.
     */
    public Algorithm getAlgorithm() {
        return this.algorithm;
    }

    /**
     * @return The JCE algorithm for {@link Signature#getInstance(String)}.
     */
    public String getJceAlgorithm() {
        
        String ret = this.algorithm.getJceName();
        
        if (ret == null && this.curve != null) {
            // this is an Edwards curve, where the curve name is an algorithm.
            ret = this.curve.getJwkType();
        }
        
        return ret;
    }
    
    /**
     * @return The allowed key operations.
     */
    public KeyOperation[] getKeyOperations() {
        return this.keyOperations;
    }

    /**
     * @return The JCE public key containing moduli or curve parameters.
     */
    public PublicKey getPublicKey() {
        return this.publicKey;
    }

    @Override
    public String toString() {
        return "PubKeyInfo [keyId=" + this.keyId + ", keyType=" + this.keyType +
                ", curve=" + this.curve + ", algorithm=" + this.algorithm +
                ", keyOperations=" + Arrays.toString(this.keyOperations) +
                ", publicKey=" + this.publicKey + "]";
    }

}
