/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2019-12-07
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/
package org.clazzes.login.jbo.common;

import java.util.HashMap;
import java.util.Map;

/**
 * A Key Type, see the
 * <a href="https://www.iana.org/assignments/cose/cose.xhtml#key-type">The COSE specification</a>
 * for details.
 */
public enum KeyType {
    
    /**
     * Octet Key Pair  [RFC8152]
     */
    OKP(1,null),
    /**
     * Elliptic Curve Keys w/ x- and y-coordinate pair [RFC8152]
     */
    EC2(2,"EC"),
    /**
     * RSA Key [RFC8230]
     */
    RSA(3,"RSA"),
    /**
     * Symmetric Keys  [RFC8152]
     */
    Symmetric(4,null);
    
    private static final Map<Integer,KeyType> typeByValue;
    private static final Map<String,KeyType> typeByJwkType;
    
    static {
        typeByValue = new HashMap<Integer,KeyType>();
        typeByJwkType = new HashMap<String,KeyType>();
        for (KeyType e : KeyType.values()) {
            typeByValue.put(e.getValue(),e);
            if (e.getJwkType() != null) {
                typeByJwkType.put(e.getJwkType(),e);
            }
        }
    }
    
    /**
     * @param value A COSE key type value.
     * @return The KeyType for the given value.
     * @throws IllegalArgumentException If the given key type value is unknown.
     */
    public static KeyType getByValue(Integer value) {
        KeyType ret = typeByValue.get(value);
        if (ret == null) {
            throw new IllegalArgumentException("The value ["+value+"] is not a known COSE key type.");
        }
        return ret;
    }
    
    /**
     * @param jwkType A JWK key type string.
     * @return The KeyType for the given JWK key type string.
     * @throws IllegalArgumentException If the given JWK key type string is unknown.
     */
    public static KeyType getByJwkType(String jwkType) {
        KeyType ret = typeByJwkType.get(jwkType);
        if (ret == null) {
            throw new IllegalArgumentException("The JWK key tyoe ["+jwkType+"] is unknown.");
        }
        return ret;
    }
    
    private final int value;
    private final String jwkType;

    private KeyType(int value, String jwkType) {
        this.value = value;
        this.jwkType = jwkType;
    }

    /**
     * @return The numeric value used as key type in COSE data structures.
     */
    public int getValue() {
        return this.value;
    }

    /**
     * @return The JWK key type, which may be <code>RSA</code> or <code>EC</code> or
     *         <code>null</code>, if this key type is not supported by JWK.
     */
    public String getJwkType() {
        return this.jwkType;
    }

}
