/***********************************************************
 * $Id$
 * 
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 28.05.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.common;

import java.util.HashMap;
import java.util.Map;

/**
 * A key operation as defined in
 * <a href="https://tools.ietf.org/html/rfc7517#section-4.3"> Section 4.2 of RFC 7515</a>.
 */
public enum KeyOperation {

    /**
     * compute digital signature or MAC
     */
    sign(1),
    /**
     * verify digital signature or MAC
     */
    verify(2),
    /**
     * encrypt content
     */
    encrypt(3),
    /**
     * decrypt content and validate decryption, if applicable
     */
    decrypt(4),
    /**
     * encrypt key
     */
    wrapKey(5),
    /**
     * decrypt key and validate decryption, if applicable
     */
    unwrapKey(6),
    /**
     * derive key
     */
    deriveKey(7),
    /**
     * derive bits (not to be used as a key)
     */
    deriveBits(8),
    /**
     * Key used to create MACs
     */
    macCreate(9),
    /**
     * Key used to verify MACs.
     */
    macVerify(10);
    
    private static final Map<Integer,KeyOperation> keyOpByValue;
    
    static {
        keyOpByValue = new HashMap<Integer,KeyOperation>();
        for (KeyOperation e : KeyOperation.values()) {
            keyOpByValue.put(e.getValue(),e);
        }
    }
    
    /**
     * @param value A key operation value as in the CBOR specification.
     * @return A key uoperation.
     * @throws IllegalArgumentException I the given value is not a defined key operation.
     */
    public static KeyOperation getByValue(Integer value) {
        KeyOperation ret = keyOpByValue.get(value);
        if (ret == null) {
            throw new IllegalArgumentException("The value ["+value+"] is not a supported KeyOperation.");
        }
        return ret;
    }

    private final int value;
    
    private KeyOperation(int value) {
        this.value = value;
    }

    public int getValue() {
        return this.value;
    }

}
