/***********************************************************
 * $Id$
 *
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 28.05.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jbo.common;

import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.util.Arrays;
import java.util.Base64;
import java.util.HexFormat;

/**
 * Static helpers for base64 et al.
 */
public class Helpers {

    /**
     * Parse base64 strings either with plus and slash signs or minus
     * and underscore signs.
     *
     * @param b64String A base64 encoded binary
     * @return The parsed byte array.
     */
    public static final byte[] parseBase64(String b64String) {

        String x = b64String.replace('-','+').replace('_','/');

        // introduce padding, because padding is sometimes omitted in JSON strings,
        // because the length is known aforehand.
        switch (x.length() % 4) {
        case 2:
            x += "==";
            break;
        case 3:
            x += "=";
        }

        return Base64.getDecoder().decode(x);
    }

    /**
     * Parse a positive {@link BigInteger} instance, which is base64 encoded.
     *
     * @param b64String A base64 encoded binary to be interpreted as a positive {@link BigInteger}
     * @return A positive integer with the magnitude of the given base64 string.
     */
    public static final BigInteger parsePositiveBigInt(String b64String) {

        return new BigInteger(1,parseBase64(b64String));
    }

    /**
     * Format a byte array as base64 with filename-save character set as
     * per <a href="https://tools.ietf.org/html/rfc4648#section-5">Section 5 of RFC 4648</a>.
     *
     * @param data An input byte array
     * @return The base 64 formatted variant without trailing equal signs and minus and
     *         underscore signs complementing alphanumeric letters.
     */
    public static final String formatBase64(byte[] data) {

        if (data == null) {
            return null;
        }

        return Base64.getUrlEncoder().encodeToString(data);
    }

    /**
     * Format a byte array as a hex string.
     *
     * @param data An input byte array
     * @return The hex formatted string.
     */
    public static final String formatHex(byte[] data) {

        if (data == null) {
            return null;
        }

        return HexFormat.of().formatHex(data);
    }


    /**
     * Format a positive big integer as base64 with filename-save character set as
     * per <a href="https://tools.ietf.org/html/rfc4648#section-5">Section 5 of RFC 4648</a>.
     *
     * For negative values a two's complement will be written.
     *
     * @param bi The big integer to write.
     * @return The base 64 formatted variant without trailing equal signs and minus and
     *         underscore signs complementing alphanumeric letters.
     */
    public static final String formatPositiveBigInt(BigInteger bi) {

        byte[] data = bi.toByteArray();

        if (data[0] == 0) {
            return formatBase64(Arrays.copyOfRange(data,1,data.length));
        }
        else {
            return formatBase64(data);
        }
    }

    /**
     * Parse a given scope value, which may contain multiple item separated by spaces.
     * May also be used for audience values.
     *
     * @param scope The scope string as submitted by an OAuth client.
     * @return The list of scope items, sorted alphabetically.
     */
    public static final String[] parseScope(String scope) {

        if (scope == null) {
            return new String[0];
        }

        String trimmed = scope.trim();

        if (trimmed.isEmpty()) {
            return new String[0];
        }

        String[] scopes = trimmed.split("\\s+");

        Arrays.sort(scopes);
        return scopes;
    }

    /**
     * Split a token composed of base64 encoded parts separated by dots.
     * @param tokenString A dot-separated token string with multiple base64 encoded parts.
     * @param limit The maximal number of parts to parse.
     * @return An array of base64-decoded byte array with at most <code>limit</code>
     *         elements.
     */
    public static final byte[][] splitToken(String tokenString, int limit) {

        String[] parts = new String[limit];
        int nparts = 0;
        int lastPos = 0;

        for (int i=0;i<tokenString.length();++i) {

            if (tokenString.charAt(i) == '.') {

                parts[nparts] = tokenString.substring(lastPos,i);

                ++nparts;

                if (nparts >= limit) {

                    throw new IllegalArgumentException("Token contains more than ["+limit+"] parts.");
                }

                lastPos = i + 1;
            }
        }

        parts[nparts] = tokenString.substring(lastPos);
        ++nparts;

        byte[][] ret = new byte[nparts][];

        for (int i=0;i<nparts;++i) {
            ret[i] = parseBase64(parts[i]);
        }

        return ret;
    }

    /**
     * @param cert A parsed certificate.
     * @return The SHA-1 hash of the encoded certificate.
     */
    public static final byte[] getSha1Fingerprint(Certificate cert) {

        try {
            MessageDigest md = MessageDigest.getInstance("SHA-1");
            md.update(cert.getEncoded());
            return md.digest();
        } catch (Exception e) {
            throw new IllegalArgumentException("Could not generate SHA-1 tumbprint for X-509 certificate.",e);
        }
    }

    /**
     * @param cert A parsed certificate.
     * @return The SHA-256 hash of the encoded certificate.
     */
    public static final byte[] getSha256Fingerprint(Certificate cert) {

        try {
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            md.update(cert.getEncoded());
            return md.digest();
        } catch (Exception e) {
            throw new IllegalArgumentException("Could not generate SHA-256 tumbprint for X-509 certificate.",e);
        }
    }

    /**
     * Compare two public keys.
     *
     * @param a A first public key.
     * @param b A second public key.
     * @return Whether the two keys match.
     */
    public static final boolean pubKeysAreEqual(PublicKey a, PublicKey b) {

        if (a == null) {
            return b == null;
        }
        else if (b==null) {
            return false;
        }

        return Arrays.equals(a.getEncoded(),b.getEncoded());
    }

}
