/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2019-12-07
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/
package org.clazzes.login.jbo.common;

import java.security.spec.ECGenParameterSpec;
import java.util.HashMap;
import java.util.Map;

/**
 * Elliptic curve type, see
 * <a href="https://www.iana.org/assignments/cose/cose.xhtml#elliptic-curves">The COSE specification</a>
 * for details.
 */
public enum CurveType {

    /**
     * EC2 NIST P-256 also known as secp256r1 [RFC8152].
     */
    P_256(1,"P-256","secp256r1","1.2.840.10045.3.1.7",Algorithm.ES256),
    /**
     * EC2 NIST P-384 also known as secp384r1 [RFC8152].
     */
    P_384(2,"P-384","secp384r1","1.3.132.0.34",Algorithm.ES384),
    /**
     * EC2 NIST P-521 also known as secp521r1 [RFC8152].
     */
    P_521(3,"P-521","secp521r1","1.3.132.0.35",Algorithm.ES512),
    /**
     * OKP X25519 for use w/ ECDH only [RFC8152].
     */
    X25519(4,"X25519",null,null,null),
    /**
     * OKP X448 for use w/ ECDH only [RFC8152].
     */
    X448(5,"X448",null,null,null),
    /**
     * OKP Ed25519 for use w/ EdDSA only [RFC8152].
     */
    Ed25519(6,"Ed25519",null,null,Algorithm.EdDSA),
    /**
     * OKP Ed448 for use w/ EdDSA only [RFC8152].
     */
    Ed448(7,"Ed448",null,null,Algorithm.EdDSA);

    private static final Map<Integer,CurveType> curvesByValue;
    private static final Map<String,CurveType> curvesByJwkType;
    private static final Map<String,CurveType> curvesByJceType;
    private static final Map<String,CurveType> curvesByOid;
    
    static {
        curvesByValue = new HashMap<Integer,CurveType>();
        curvesByJwkType = new HashMap<String,CurveType>();
        curvesByJceType = new HashMap<String,CurveType>();
        curvesByOid = new HashMap<String,CurveType>();
        for (CurveType e : CurveType.values()) {
            curvesByValue.put(e.getValue(),e);
            curvesByJwkType.put(e.getJwkType(),e);
            if (e.getOid() != null) {
                curvesByOid.put(e.getOid(),e);
            }
            if (e.getJceType() != null) {
                curvesByJceType.put(e.getJceType(),e);
            }
        }
    }
    
    /**
     * @param value A COSE elliptic curve type value.
     * @return The CurveType for the given value.
     * @throws IllegalArgumentException If the given elliptic curve type value is unknown.
     */
    public static CurveType getByValue(Integer value) {
        CurveType ret = curvesByValue.get(value);
        if (ret == null) {
            throw new IllegalArgumentException("The value ["+value+"] is not a known COSE Elliptic Curve.");
        }
        return ret;
    }
    
    /**
     * @param jwkType A JWK elliptic curve name.
     * @return The CurveType for the given JWK curve name.
     * @throws IllegalArgumentException If the given JWK curve name is unknown.
     */
    public static CurveType getByJwkType(String jwkType) {
        CurveType ret = curvesByJwkType.get(jwkType);
        if (ret == null) {
            throw new IllegalArgumentException("The JWK name ["+jwkType+"] is not a known Elliptic Curve.");
        }
        return ret;
    }

    /**
     * @param jceType A JCE elliptic curve name.
     * @return The CurveType for the given JCE curve name.
     * @throws IllegalArgumentException If the given JCE curve name is unknown.
     */
    public static CurveType getByJceType(String jceType) {
        CurveType ret = curvesByJceType.get(jceType);
        if (ret == null) {
            throw new IllegalArgumentException("The JCE name ["+jceType+"] is not a known Elliptic Curve.");
        }
        return ret;
    }

    /**
     * @param oid An elliptic curve OID.
     * @return The CurveType for the given OID.
     * @throws IllegalArgumentException If the given OID is unknown.
     */
    public static CurveType getByOid(String oid) {
        CurveType ret = curvesByOid.get(oid);
        if (ret == null) {
            throw new IllegalArgumentException("The OID ["+oid+"] is not a known Elliptic Curve.");
        }
        return ret;
    }

    private final int value;
    private final String jwkType;
    private final String jceType;
    private final String oid;
    private final Algorithm algorithm;

    private CurveType(int value, String jwkType, String jceType, String oid, Algorithm algorithm) {
        this.value = value;
        this.jwkType = jwkType;
        this.jceType = jceType;
        this.oid = oid;
        this.algorithm = algorithm;
    }

    /**
     * @return The numeric value used as curve type in COSE data structures.
     */
    public int getValue() {
        return this.value;
    }

    /**
     * @return The JWK curve type string
     */
    public String getJwkType() {
        return this.jwkType;
    }

    /**
     * @return The curve name as accepted by {@link ECGenParameterSpec}.
     */
    public String getJceType() {
        return this.jceType;
    }

    /**
     * @return The OID of this curve, if this curve is referenced as an algorithm parameter.
     */
    public String getOid() {
        return this.oid;
    }

    /**
     * @return The algorithm used with this key or <code>null</code>, of this key does not
     *         have a preferred algorithm to use.
     */
    public Algorithm getAlgorithm() {
        return this.algorithm;
    }
  
}
