/***********************************************************
 * $Id$
 * 
 * JSON CBOR Login Tools of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 08.12.2019
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.jbo.common;

import java.util.HashMap;
import java.util.Map;

/**
 * Cryptographic algorithms, see
 * <a href="https://www.iana.org/assignments/cose/cose.xhtml#algorithms">The COSE specification</a>
 * for details.
 * 
 * This class covers only supported signature algorithms.
 */
public enum Algorithm {

    /**
     * RSASSA-PKCS1-v1_5 w/ SHA-1      [draft-jones-webauthn-cose-algorithms]
     */
    RS1(-65535,"SHA1withRSA","SHA-1",KeyType.RSA),
    /**
     * RSASSA-PKCS1-v1_5 w/ SHA-512    [draft-jones-webauthn-cose-algorithms]
     */
    RS512(-259,"SHA512withRSA","SHA-512",KeyType.RSA),
    /**
     * RSASSA-PKCS1-v1_5 w/ SHA-384    [draft-jones-webauthn-cose-algorithms]
     */
    RS384(-258,"SHA384withRSA","SHA-384",KeyType.RSA),
    /**
     * RSASSA-PKCS1-v1_5 w/ SHA-256    [draft-jones-webauthn-cose-algorithms]
     */
    RS256(-257,"SHA256withRSA","SHA-256",KeyType.RSA),
    /**
     * RSAES-OAEP w/ SHA-512   [RFC8230]
     */
    RSAES_OAEP_SHA_512(-42,"OAEPWithSHA-512AndMGF1Padding","SHA-512",KeyType.RSA),
    /**
     * RSAES-OAEP w/ SHA-256   [RFC8230]
     */
    RSAES_OAEP_SHA_256(-41,"OAEPWithSHA-256AndMGF1Padding","SHA-256",KeyType.RSA),
    /**
     * RSAES-OAEP w/ SHA-1 and RFC 8017 default parameters    [RFC8230]
     */
    RSAES_OAEP_SHA_1(-40,"OAEPWithSHA-1AndMGF1Padding","SHA-1",KeyType.RSA),
    /**
     * RSASSA-PSS w/ SHA-512   [RFC8230]
     */
    PS512(-39,"SHA512withRSAandMGF1","SHA-512",KeyType.RSA),
    /**
     * RSASSA-PSS w/ SHA-384   [RFC8230]
     */
    PS384(-38,"SHA384withRSAandMGF1","SHA-384",KeyType.RSA),
    /**
     * RSASSA-PSS w/ SHA-256   [RFC8230]
     */
    PS256(-37,"SHA256withRSAandMGF1","SHA-256",KeyType.RSA),
    /**
     * ECDSA w/ SHA-512        [RFC8152]
     */
    ES512(-36,"SHA512withECDSA","SHA-512",KeyType.EC2),
    /**
     * ECDSA w/ SHA-384        [RFC8152]
     */
    ES384(-35,"SHA384withECDSA","SHA-384",KeyType.EC2),
    /**
     * ECDSA w/ SHA-256        [RFC8152]
     */
    ES256(-7,"SHA256withECDSA","SHA-256",KeyType.EC2),
    /**
     * EdDSA   [RFC8152]
     */
    EdDSA(-8,null,null,KeyType.OKP);

    private static final Map<Integer,Algorithm> algosByValue;
    private static final Map<String,Algorithm> algosByJceName;
    
    static {
        algosByValue = new HashMap<Integer,Algorithm>();
        algosByJceName = new HashMap<String, Algorithm>();
        for (Algorithm e : Algorithm.values()) {
            algosByValue.put(e.getValue(),e);
            if (e.getJceName() != null) {
                algosByJceName.put(e.getJceName(),e);
            }
        }
    }
    
    /**
     * @param value An integer JOSE algorithm identifier.
     * @return The algorithm.
     * @throws IllegalArgumentException If the given algorithm identifier is not known.
     */
    public static Algorithm getByValue(Integer value) {
        Algorithm ret = algosByValue.get(value);
        if (ret == null) {
            throw new IllegalArgumentException("The value ["+value+"] is not a supported COSE cryptographic algorithm.");
        }
        return ret;
    }
    
    private final int value;
    private final String jceName;
    private final String hashName;
    private final KeyType keyType;
    
    private Algorithm(int value,String jceName,String hashName,KeyType keyType) {
        this.value = value;
        this.jceName = jceName;
        this.hashName = hashName;
        this.keyType = keyType;
    }

    /**
     * @return The algorithm identifier as used in COSE data structures. 
     */
    public int getValue() {
        return this.value;
    }

    /**
     * @return The JCE algorithm name.
     */
    public String getJceName() {
        return this.jceName;
    }

    /**
     * @return The standard hash associated with this algorithm or <code>null</code>, if this
     *         algorithm has no associated default hash.
     */
    public String getHashName() {
        return this.hashName;
    }

    /**
     * @return The type of key used for this algorithm.
     */
    public KeyType getKeyType() {
        return this.keyType;
    }
    
}
