/***********************************************************
 *
 * Login Components of
 *
 * https://www.clazzes.org
 *
 * Created: 04.04.2025
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jaas;

import java.nio.file.Path;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.clazzes.login.htpasswd.jaas.HtpasswdAuthServiceFactoryConfiguration;
import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.util.sec.DomainPasswordLoginService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JaasLoginComponent implements Component {

    private static final Logger log = LoggerFactory.getLogger(JaasLoginComponent.class);

    public static final String CONFIG_PID = "org.clazzes.login.jaas";
    public static final String SERVICE_KEY = "org.clazzes.login.jaas";

    private static final HtpasswdAuthServiceFactoryConfiguration parseConfiguration(ConfigWrapper config) {
        var htpasswdDirString = config.getString("htpasswdDir");

        return new HtpasswdAuthServiceFactoryConfiguration()
            // Will fall back to system properties if config is not specified.
            .htpasswdDir(htpasswdDirString == null ? null : Path.of(htpasswdDirString))
            .htpasswdFileContents(Optional.ofNullable(config.getSubTree("htpasswdFileContents"))
                .map(subConfig -> subConfig.getContent()
                    .entrySet()
                    .stream()
                    .map(entry -> Map.entry(entry.getKey(), (String) entry.getValue()))
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)))
                .orElse(null));
    }

    @Override
    public void start(ServiceContext context) throws Exception {

        ConfigurationEngine ce = context.getService(ConfigurationEngine.class).get();
        ServiceRegistry registry = context.getService(ServiceRegistry.class).get();
        ComponentManager componentManager = context.getService(ComponentManager.class).get();

        var initConfig = ce.getPid(CONFIG_PID)
            .map(c -> parseConfiguration(c))
            .orElse(new HtpasswdAuthServiceFactoryConfiguration());

        var htpasswdFactory = initConfig.build();

        JaasDomainPasswordLoginService jdps = new JaasDomainPasswordLoginService();
        jdps.setHtpasswdFactory(htpasswdFactory);

        registry.addService(SERVICE_KEY,DomainPasswordLoginService.class,jdps);


        ce.listen(CONFIG_PID,config -> { synchronized(this) {
            jdps.setDefaultDomain(config.getString("defaultDomain"));
            htpasswdFactory.setConfig(parseConfiguration(config));
        }});

        // commit here, to be up even without a configured domain.
        componentManager.commit();
    }

    @Override
    public void stop(ServiceContext context) throws Exception {

        ServiceRegistry registry = context.getService(ServiceRegistry.class).get();
        registry.removeService(SERVICE_KEY,DomainPasswordLoginService.class);
    }

}
