/***********************************************************
 * $Id$
 *
 * http://www.clazzes.org
 *
 * Created: 02.04.2011
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.jaas;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;

import org.clazzes.login.htpasswd.jaas.HtpasswdAuthServiceFactory;
import org.clazzes.login.htpasswd.jaas.HtpasswdGroup;
import org.clazzes.login.htpasswd.jaas.IHtpasswdAuthService;
import org.clazzes.util.sec.DomainGroup;
import org.clazzes.util.sec.DomainPasswordLoginService;
import org.clazzes.util.sec.DomainPrincipal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JaasDomainPasswordLoginService implements DomainPasswordLoginService {

    private static final Logger log = LoggerFactory.getLogger(JaasDomainPasswordLoginService.class);

    private String defaultDomain;
    private HtpasswdAuthServiceFactory htpasswdFactory;
    private final Map<JaasPrincipal,List<DomainGroup>> userGroups = new ConcurrentHashMap<JaasPrincipal, List<DomainGroup>>();

    @Override
    public DomainPrincipal tryLogin(String domain, String username, String password) {

        try {
            IHtpasswdAuthService svc = htpasswdFactory.getServiceConfigurable(domain);

            if (!svc.authenticate(username,password.toCharArray())) {
                return null;
            }

            List<HtpasswdGroup> htpasswdGroups = svc.getUserGroups(username);
            List<DomainGroup> groups = null;

            if (htpasswdGroups != null) {

                groups = new ArrayList<DomainGroup>(htpasswdGroups.size());
                for (HtpasswdGroup grp:htpasswdGroups) {
                    groups.add(new JaasGroup(grp.getName(),domain));
                }
            }

            Map<String,String> additionalAttributes = svc.getUserClaims(username);

            JaasPrincipal ret;

            if (additionalAttributes == null) {

                ret = new JaasPrincipal(username,domain);

                if (groups != null) {
                    log.info("User [{}] is member of groups {}.",
                            username,groups);
                    this.userGroups.put(ret,groups);
                }

                log.info("Successful login of user [{}] to domain [{}].",
                        new Object[]{username,domain});
            }
            else {

                ret = new JaasPrincipalOAuth(username,domain,
                            Collections.unmodifiableMap(additionalAttributes));

                log.info("Successful login of user [{}] to domain [{}] with additional attributes [{}].",
                        new Object[]{username,domain,additionalAttributes});

                if (groups != null) {
                    log.info("User [{}] is member of groups {}.",
                            username,groups);
                    // use plain JaasPrincipal as key in order to fix
                    // the follow up of #18
                    this.userGroups.put(new JaasPrincipal(username,domain),groups);
                }
            }
            return ret;

        } catch (Exception e) {

            if (log.isDebugEnabled()) {
                log.debug("Caught exception during JAAS authentication",e);
            }

            log.error("Invalid login of user [{}] to domain [{}].",
                      username,domain);

            return null;
        }
    }


    @Override
    public synchronized String getDefaultDomain() {
        return this.defaultDomain;
    }

    //@Override
    public List<String> getDomains() {
        Vector<String> domains = new Vector<String>();
        String defaultDomain = this.getDefaultDomain();
        if (defaultDomain != null && defaultDomain.length() > 0) {
            domains.add(defaultDomain);
        }
        return domains;
    }


    @Override
    public void changePassword(String domain, String username, String oldPassword, String newPassword) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void deactivateUser(String domain, String username, String reason) {
        throw new UnsupportedOperationException();
    }

    @Override
    public List<DomainGroup> getGroups(String domain, String userName) {

        return this.userGroups.get(new JaasPrincipal(userName,domain));
    }

    @Override
    public List<DomainPrincipal> getGroupMembers(String domain, String groupName) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int getSupportedFeatures(String domain) {
        return DomainPasswordLoginService.FEATURE_GET_GROUPS;
    }

    @Override
    public DomainPrincipal searchUser(String domain, String username) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void sendPassword(String domain, String username) {
        throw new UnsupportedOperationException();
    }

    public synchronized void setDefaultDomain(String defaultDomain) {
        this.defaultDomain = defaultDomain;
    }

    public void setHtpasswdFactory(HtpasswdAuthServiceFactory htpasswdFactory) {
        this.htpasswdFactory = htpasswdFactory;
    }
}
