/***********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.adapter.http;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TimeZone;

import org.clazzes.util.sec.DomainPrincipal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A cache for login infos.
 */
public class LoginInfoCache {

	private static final Logger log = LoggerFactory.getLogger(LoginInfoCache.class);

	private final Map<String,LoginInfo> infosBySessionId;
	private TokenGenerator tokenGenerator;

	public LoginInfoCache() {

		this.infosBySessionId = new HashMap<String, LoginInfo>(1024);

	}

	public synchronized LoginInfo getLoginInfo(String sessionId) {

		LoginInfo ret = this.infosBySessionId.get(sessionId);

		return ret;
	}

	public synchronized LoginInfo removeLoginInfo(String sessionId) {

		return this.infosBySessionId.remove(sessionId);
	}


	public LoginInfo createLoginInfo(String sessionId, String mechanism,
	                                 MFAState mfaState,
									 Locale locale, TimeZone timeZone,
									 long maxAge,
									 boolean fromOAuth) {

		LoginInfo ret = null;

		if (sessionId != null) {
			synchronized(this) {

				ret = this.infosBySessionId.get(sessionId);
			}
		}

		if (ret != null) {

            if (locale != null && !locale.equals(ret.getLocale())) {
                if (log.isWarnEnabled()) {
                    log.warn("Overriding locale [{}] of login [{}] with new value [{}] from mechanism [{}].",
                            new Object[]{ret.getLocale(),ret.getPrincipalsInfo(),locale,mechanism});
                }
                ret.setLocale(locale);
            }

            if (timeZone != null && !timeZone.equals(ret.getTimeZone())) {

                if (log.isWarnEnabled()) {
                    log.warn("Overriding time zone [{}] of login [{}] with new value [{}] from mechanism [{}].",
                            new Object[]{ret.getTimeZone(),ret.getPrincipalsInfo(),timeZone,mechanism});
                }
                ret.setTimeZone(timeZone);
            }

            ret.addMFAState(mechanism,mfaState);
            ret.touch(maxAge);
            return ret;
        }

		int ntry = 0;

		DomainPrincipal principal = mfaState.getPrincipal();;

		do {
			++ntry;

			String key = this.tokenGenerator.generateToken();

			ret = new LoginInfo(key,locale,timeZone,fromOAuth);

			synchronized(this) {

				if (this.infosBySessionId.containsKey(key)) {
					log.warn("Duplicate session ID generated by SecureRandom for principal [{}] of type [{}].",
							principal.getName(),
							principal.getClass().getName());

					ret = null;
				}
				else {
					this.infosBySessionId.put(key,ret);
					ret.addMFAState(mechanism,mfaState);
				}
			}

		} while (ret == null && ntry < 5);

		if (ret == null) {
			throw new SecurityException("["+ntry+
					"] duplicate session IDs generated by SecureRandom for principal ["+principal.getName()+
					"] of type ["+principal.getClass().getName()+"].");
		}

		ret.touch(maxAge);
		return ret;
	}

	public void gc() {

		long now = System.currentTimeMillis();

		synchronized (this) {

			if (log.isDebugEnabled()) {
				log.debug("Starting login info garbage collection, number of persisted session IDs is [{}]...",
						LoginInfoCache.this.infosBySessionId.size());
			}

			Iterator<Map.Entry<String,LoginInfo>> it = LoginInfoCache.this.infosBySessionId.entrySet().iterator();

			while (it.hasNext()) {

				Entry<String, LoginInfo> e = it.next();

				if (e.getValue().getExpires() <= now) {

					if (log.isWarnEnabled()) {

						log.warn("Login [{}] expired without prior logout.",
								e.getValue().getPrincipalsInfo());
					}

					it.remove();
				}
			}

			if (log.isDebugEnabled()) {
				log.debug("Login info garbage collection finished, number of persisted session IDs is [{}].",
						LoginInfoCache.this.infosBySessionId.size());
			}
		}
	}

    public void setTokenGenerator(TokenGenerator tokenGenerator) {
        this.tokenGenerator = tokenGenerator;
    }

}
