/***********************************************************
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 07.04.2025
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.adapter.http;

import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ComponentSupport;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.CoreService;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.util.http.sec.PageTokenService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpLoginComponent extends ComponentSupport implements Component {

    private static final Logger log = LoggerFactory.getLogger(HttpLoginComponent.class);

    public static final String CONFIG_PID = "org.clazzes.login.adapter.http";

    private GarbageCollector garbageCollector;
    private LoginServiceListener loginServiceListener;

    @Override
    public synchronized void start(ServiceContext context) throws Exception {

        CoreService coreService = context.getService(CoreService.class).get();
        ServiceRegistry serviceRegistry = context.getService(ServiceRegistry.class).get();
        ConfigurationEngine configurationEngine = context.getService(ConfigurationEngine.class).get();
        ComponentManager componentManager = context.getService(ComponentManager.class).get();

        TokenGenerator tg = new TokenGenerator();

        PageTokenServiceImpl ptsi = new PageTokenServiceImpl();
        ptsi.setTokenGenerator(tg);

        this.addService(serviceRegistry,"http-login",PageTokenService.class,ptsi);

        LoginInfoCache lic = new LoginInfoCache();
        lic.setTokenGenerator(tg);


        this.loginServiceListener = new LoginServiceListener(serviceRegistry,componentManager);

        this.loginServiceListener.setLoginInfoCache(lic);
        this.loginServiceListener.setPageTokenService(ptsi);

        this.addListener(configurationEngine.listen(CONFIG_PID,this.loginServiceListener));

        this.garbageCollector = new GarbageCollector();
        this.garbageCollector.setPageTokenService(ptsi);
        this.garbageCollector.setLoginInfoCache(lic);

        this.garbageCollector.executorServiceBound(coreService.getScheduledExecutorService());
    }

    @Override
    public synchronized void stop(ServiceContext context) throws Exception {

        CoreService coreService = context.getService(CoreService.class).get();
        ServiceRegistry serviceRegistry = context.getService(ServiceRegistry.class).get();

        this.removeAllServices(serviceRegistry);
        this.closeAllListeners((c,e) -> {
            log.warn("Error closing [{}]",c,e);
        });

        if (this.garbageCollector != null) {
            this.garbageCollector.executorServiceUnbound(coreService.getScheduledExecutorService());
            this.garbageCollector = null;
        }

        if (this.loginServiceListener != null) {
            this.loginServiceListener.stop();
            this.loginServiceListener = null;
        }

    }

}
