/***********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.adapter.http;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ConfigurationHelper;
import org.clazzes.util.http.RequestHelper;
import org.clazzes.util.http.sec.PageTokenService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.servlet.http.HttpServletRequest;

/**
 * A counter-XSRF page token service.
 */
public class PageTokenServiceImpl implements PageTokenService, Consumer<ConfigWrapper> {

    private static final Logger log = LoggerFactory.getLogger(PageTokenServiceImpl.class);

	private static final int DEFAULT_SESSION_TIMEOUT = 180;
	private static final int DEFAULT_MAX_PAGE_TOKENS = 5000;

    private final AtomicInteger sessionTimeout;
    private final AtomicInteger maxPageTokens;
    private TokenGenerator tokenGenerator;

    private final ConcurrentMap<String,Long> pageTokens;

    public PageTokenServiceImpl() {
        this.sessionTimeout = new AtomicInteger(DEFAULT_SESSION_TIMEOUT);
        this.maxPageTokens = new AtomicInteger(DEFAULT_MAX_PAGE_TOKENS);
        this.pageTokens = new ConcurrentHashMap<String,Long>();
    }

    private Long makeExpirationTimestamp() {
        return Long.valueOf(System.currentTimeMillis()+this.sessionTimeout.get()*60000L);
    }

	/* (non-Javadoc)
	 * @see org.clazzes.util.http.sec.PageTokenService#getPageToken(javax.servlet.http.HttpServletRequest)
	 */
	@Override
	public String getPageToken(HttpServletRequest req) {

	    String token = this.tokenGenerator.generateToken();

	    if (log.isDebugEnabled()) {

	        log.debug("Generated page token [{}] for client [{}].",token,RequestHelper.getRealRemoteIP(req));
	    }

	    this.pageTokens.put(token,this.makeExpirationTimestamp());

		return token;
	}

	/* (non-Javadoc)
	 * @see org.clazzes.util.http.sec.PageTokenService#checkPageToken(javax.servlet.http.HttpServletRequest, java.lang.String)
	 */
	@Override
	public boolean checkPageToken(HttpServletRequest req, String token) {

	    boolean ret = this.pageTokens.containsKey(token);

	    if (ret) {

	        this.pageTokens.put(token,this.makeExpirationTimestamp());

	        if (log.isDebugEnabled()) {

	            log.debug("Successfully verified page token [{}] for client [{}].",token,RequestHelper.getRealRemoteIP(req));
	        }
	    }
	    else {

	        if (log.isWarnEnabled()) {

	            log.warn("Received invalid page token [{}] from client [{}].",token,RequestHelper.getRealRemoteIP(req));
	        }
	    }

	    return ret;
	}

	public void gc() {

	    int ntokens = this.pageTokens.size();

	    if (log.isDebugEnabled()) {
	        log.debug("Garbage collecting [{}] page tokens...",ntokens);
	    }

	    Queue<Entry<String, Long>> watermarkQueue = null;
	    List<String> tokensToDelete = null;

		int maxTokens = this.maxPageTokens.get();

	    if (ntokens > maxTokens) {

            if (log.isWarnEnabled()) {

                 log.warn("The maximal number of [{}] page tokens has been exceeded, prematurely deleting tokens.",
                        this.maxPageTokens);
            }

	        tokensToDelete = new ArrayList<String>(10+ntokens-maxTokens);

	        // insert entries in a priority queue as per expiration date.
	        watermarkQueue = new PriorityQueue<Entry<String, Long>>(
				maxTokens+1, new Comparator<Entry<String, Long>>() {

                        @Override
                        public int compare(Entry<String, Long> o1,
                                Entry<String, Long> o2) {
                            return o1.getValue().compareTo(o2.getValue());
                        }
                    });
	    }

	    long now = System.currentTimeMillis();

	    Iterator<Entry<String, Long>> it = this.pageTokens.entrySet().iterator();

	    while (it.hasNext()) {

	        Entry<String, Long> e = it.next();

	        if (e.getValue() <= now) {

	            if (log.isDebugEnabled()) {
	                log.debug("Page token [{}] has expired.",e.getKey());
	            }

	            it.remove();
	        }
	        else if (watermarkQueue != null){

	            watermarkQueue.add(e);
	            if (watermarkQueue.size() > maxTokens) {

	                Entry<String, Long> ed = watermarkQueue.poll();

	                if (log.isWarnEnabled()) {

	                    long delta = ed.getValue()-now;

	                    log.warn(String.format(Locale.ENGLISH,"Page token [%s] deleted [%dh%d'%.3f\"] before expiry.",
	                            ed.getKey(),delta/3600000L,(delta/60000L)%60L,(delta%60000L)*0.001));
	                }

	                tokensToDelete.add(ed.getKey());
	            }
	        }
	    }

	    if (tokensToDelete != null) {

	        for (String token : tokensToDelete) {
	            this.pageTokens.remove(token);
	        }
	    }

	    if (log.isDebugEnabled()) {
            log.debug("Finished page token garbage collection with [{}] remaining page tokens.",this.pageTokens.size());
        }
	}

    @Override
    public void accept(ConfigWrapper properties) {

        if (log.isDebugEnabled()) {
            log.debug("updated called: properties=[{}]",properties);
        }

		this.sessionTimeout.set(properties.getInt("sessionTimeout",DEFAULT_SESSION_TIMEOUT));

		this.maxPageTokens.set(properties.getInt("maxPageTokens",DEFAULT_MAX_PAGE_TOKENS));
    }

    public void setTokenGenerator(TokenGenerator tokenGenerator) {
        this.tokenGenerator = tokenGenerator;
    }

}
