/***********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.adapter.http;

import java.net.URI;
import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.http.LocaleHelper;
import org.clazzes.util.http.RequestHelper;
import org.clazzes.util.http.UrlHelper;
import org.clazzes.util.http.sec.HttpLoginService;
import org.clazzes.util.sec.DomainGroup;
import org.clazzes.util.sec.DomainPasswordLoginService;
import org.clazzes.util.sec.DomainPrincipal;
import org.clazzes.util.sec.MFAPrincipal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>An implementation of a {@link HttpLoginService}, which provides
 * HTTP session management for a given {@link DomainPasswordLoginService}</p>
 *
 * <p>This service binds the HTTP locale to the current thread using
 * {@link ThreadLocalManager#bindLoginLocale(Locale)} in order to allow
 * {@link DomainPasswordLoginService} implementations to obtain the user's
 * language when called from a HTTP context.
 * </p>
 */
public class DomainHttpLoginService implements HttpLoginService {

    private static final Logger log = LoggerFactory.getLogger(DomainHttpLoginService.class);

    private DomainPasswordLoginService domainPasswordLoginService;
    private MFAService mfaService;
    private String loginMechanism;
    private int sessionTimeout;
    private long failureTimeout;
    private boolean doTimeZoneDetection;
    private String sessionCookie;
    private boolean secureCookie;
    private SameSitePolicy sameSitePolicy;
    private boolean logoutAllMechanisms;
    private boolean doGroupsCheck;
    private int ephemeralOtpSeconds;

    private String loginUrl;

    private LoginInfoCache loginInfoCache;

    public DomainHttpLoginService() {
        this.logoutAllMechanisms = true;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.http.sec.HttpLoginService#getLoginUrl()
     */
    @Override
    public String getLoginUrl() {

        return this.loginUrl;
    }

    private final String parseCookie(HttpServletRequest req) {

        String allCookies_s = req.getHeader("Cookie");

        if (allCookies_s == null) {
            return null;
        }

        String[] allCookies = allCookies_s.split("\\s*;\\s*");

        for (String s: allCookies) {

            String[] kv = s.split("\\s*=\\s*",2);

            if (kv.length == 2 && kv[0].equals(this.sessionCookie)) {
                return kv[1];
            }
        }

        return null;
    }

    private LoginInfo getLoginInfoFromCookie(HttpServletRequest req) {

        String sessionId = this.parseCookie(req);

        if (sessionId == null) {
            return null;
        }
        else {
            return this.loginInfoCache.getLoginInfo(sessionId);
        }
    }

    @Override
    public Principal checkLogin(HttpServletRequest req) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo == null) return null;

        MFAState state = loginInfo.getMFAState(this.loginMechanism);

        if (state == null || state.getState() != MFAState.State.AUTHENTICATED) {
            return null;
        }

        Principal ret = state.getPrincipal();

        if (ret != null) {
            loginInfo.touch(this.sessionTimeout * 60000L);
        }

        return ret;
    }



    /* (non-Javadoc)
     * @see org.clazzes.util.http.sec.HttpLoginService#checkLoginGroups(javax.servlet.http.HttpServletRequest)
     */
    @Override
    public List<? extends Principal> checkLoginGroups(HttpServletRequest req) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo == null) return null;

        MFAState state = loginInfo.getMFAState(this.loginMechanism);

        if (state == null || state.getState() != MFAState.State.AUTHENTICATED) {
            return null;
        }

        return state.getGroups();
    }

    public MFAState checkMFALogin(HttpServletRequest req) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo == null) return null;

        MFAState state = loginInfo.getMFAState(this.loginMechanism);

        if (state != null) {
            loginInfo.touch(this.sessionTimeout * 60000L);
        }

        return state;
    }

    @Override
    public Locale getLocale(HttpServletRequest req) {

        Locale loc = null;
        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo != null) {
            // valid cookie
            loc = loginInfo.getLocale();
        }

        // fallback 1, query parameter, see LOGIN-30
        String loc_s = UrlHelper.getStringParameter(req.getQueryString(),"locale");
        if (loc_s != null) {
            loc = LocaleHelper.localeFromXsLanguage(loc_s);
        }

        // fallback 2
        if (loc == null) {
            loc = req.getLocale();
        }

        // fallback 3
        if (loc == null) {
            // should be unreachable, but if a request implementation fails to return a Locale this at least fulfills this method's javadoc
            loc = Locale.getDefault();
        }

        return loc;
    }

    @Override
    public TimeZone getTimeZone(HttpServletRequest req) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        TimeZone tz = null;

        if (loginInfo != null) {
            tz = loginInfo.getTimeZone();
        }

        if (tz == null) {
            // This is most likely reached, when loginInfo.getTimeZone() returned null.
            tz = TimeZone.getDefault();
        }

        return tz;
    }

    @Override
    public boolean checkPermission(HttpServletRequest req, String context) {

        return context.equals(this.loginUrl);
    }

    @Override
    public void logout(HttpServletRequest req) {

        this.logoutCheckForOAuth(req);
    }

    public boolean logoutCheckForOAuth(HttpServletRequest req) {

        String sessionId = this.parseCookie(req);

        if (sessionId == null) {
            return false;
        }

        if (this.logoutAllMechanisms) {

            LoginInfo loginInfo = this.loginInfoCache.removeLoginInfo(sessionId);

            if (loginInfo != null) {

                log.info("Logout of [{}] from all mechanisms.",loginInfo.getPrincipalsInfo());
                return loginInfo.isFromOAuth();
            }

            return false;
        }
        else {

            LoginInfo loginInfo = this.loginInfoCache.getLoginInfo(sessionId);

            if (loginInfo != null) {

                log.info("Logout of [{}] from mechanism [{}].",
                        loginInfo.getPrincipalsInfo(),this.loginMechanism);

                MFAState state = loginInfo.removeMFAState(this.loginMechanism);

                if (state == null) {
                    log.info("User [{}] is areay logged out from mechanism [{}].",
                            loginInfo.getPrincipalsInfo(),this.loginMechanism);
                }

                return loginInfo.isFromOAuth();
            }

            return false;
        }
    }

    public String getDefaultDomain() {
        return this.domainPasswordLoginService.getDefaultDomain();
    }

    public List<String> getDomains() {
        return this.domainPasswordLoginService.getDomains();
    }

    public MFAState checkTokenOtp(HttpServletRequest req, HttpServletResponse resp,
            String otp, Locale locale) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo == null) {
            return null;
        }

        MFAState state = loginInfo.getMFAState(this.loginMechanism);

        if (state == null) {
            return null;
        }

        MFAPrincipal mfaPrincipal = state.getMfaPrincipal();

        if (mfaPrincipal == null) {
            return state;
        }

        log.info("Checking token OTP for user [{}]...",state.getPrincipal().getName());

        boolean ok = this.getMfaService().checkTokenOtp(otp,mfaPrincipal.getKnownTokenIds());

        log.info("Check of token OTP for user [{}] returned [{}].",state.getPrincipal().getName(),ok);

        if (ok) {

            MFAState ret = new MFAState(state);

            loginInfo.addMFAState(this.loginMechanism,ret);

            return ret;
        }
        else {
            return state;
        }
    }

    public MFAState checkEphemeralOtp(HttpServletRequest req, HttpServletResponse resp,
            String otp, Locale locale) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo == null) {
            return null;
        }

        MFAState state = loginInfo.getMFAState(this.loginMechanism);

        if (state == null) {
            return null;
        }

        MFAPrincipal mfaPrincipal = state.getMfaPrincipal();

        if (mfaPrincipal == null) {
            return state;
        }

        // check expiry timestamp of ephemeral OTP.
        if (System.currentTimeMillis() > state.getEphemeralOtpExpiry()) {

            log.error("Ephemeral OTP for user [{}] has already expired.",state.getPrincipal().getName());
            return null;
        }

        log.info("Checking ephemeral OTP for user [{}]...",state.getPrincipal().getName());

        boolean ok = otp != null && otp.equals(state.getEphemeralOtp());

        log.info("Check of ephemeral OTP for user [{}] returned [{}].",state.getPrincipal().getName(),ok);

        if (ok) {

            MFAState ret = new MFAState(state);

            loginInfo.addMFAState(this.loginMechanism,ret);

            return ret;
        }
        else {
            return state;
        }
    }

    public MFAState generateSmsToken(HttpServletRequest req, Locale locale) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo == null) {
            return null;
        }

        MFAState state = loginInfo.getMFAState(this.loginMechanism);

        if (state == null) {
            return null;
        }

        DomainPrincipal principal = state.getPrincipal();
        List<? extends Principal> groups = state.getGroups();
        MFAPrincipal mfaPrincipal = state.getMfaPrincipal();

        if (mfaPrincipal == null) {
            return state;
        }

        String host;
        try {
            URI uri = RequestHelper.getOriginalRequestUri(req);
            host = uri.getHost();

            log.info("Generating OTP for user [{}] for original URI [{}].",principal.getName(),uri);

        } catch (Exception e) {

            log.error("Unable to determine original request URI, generating OTP for user ["+principal.getName()+"] unknown host.",e);
            host = "<unkown>";
        }

        String otp = this.getMfaService().generateEmphemeralOtp(locale,host,principal,mfaPrincipal);

        MFAState ret = new MFAState(principal,groups,otp,System.currentTimeMillis()+this.ephemeralOtpSeconds*1000L);

        loginInfo.addMFAState(this.loginMechanism,ret);

        return ret;
    }

    public MFAState tryLogin(HttpServletRequest req, HttpServletResponse resp, String domain,
            String user, String password, Locale locale, TimeZone tz) {

        if (locale != null) {
            ThreadLocalManager.bindLoginLocale(locale);
        }

        DomainPrincipal principal;
        List<DomainGroup> groups = null;

        try {

            if (log.isDebugEnabled()) {
                log.debug("Checking password of user [{}] with domain [{}]...",user,domain);
            }

            principal = this.domainPasswordLoginService.tryLogin(domain,user,password);

            if (principal != null) {
                if (log.isDebugEnabled()) {
                    log.debug("Successfully logged on as [{}].",principal.getName());
                }

                if (this.isDoGroupsCheck() &&
                        (this.domainPasswordLoginService.getSupportedFeatures(domain) & DomainPasswordLoginService.FEATURE_GET_GROUPS) != 0) {


                    if (log.isDebugEnabled()) {
                        log.debug("Checking groups of user [{}] with domain [{}]...",user,domain);
                    }

                    groups = this.domainPasswordLoginService.getGroups(domain,user);

                    if (log.isDebugEnabled()) {
                        log.debug("Successfully checked groups as [{}].",groups);
                    }
                }
            }
        }
        finally {
            if (locale != null) {
                ThreadLocalManager.unbindLoginLocale();
            }
        }

        if (principal == null) {

            log.error("Invalid initial login of user [{}] to domain [{}].",user,domain);
            return null;
        }
        else {

            String sessionId = this.parseCookie(req);

            MFAState mfaState = new MFAState(principal,groups);

            MFAPrincipal mfaPrincipal = mfaState.getMfaPrincipal();

            if (mfaPrincipal != null && mfaPrincipal.getKnownTokenIds() == null) {

                String host;
                try {
                    URI uri = RequestHelper.getOriginalRequestUri(req);
                    host = uri.getHost();

                    log.info("Generating OTP for user [{}] for original URI [{}].",principal.getName(),uri);

                } catch (Exception e) {

                    log.error("Unable to determine original request URI, generating OTP for user ["+principal.getName()+"] unknown host.",e);
                    host = "<unkown>";
                }

                String otp = this.getMfaService().generateEmphemeralOtp(locale,host,principal,mfaPrincipal);

                mfaState = new MFAState(principal,groups,otp,System.currentTimeMillis()+this.ephemeralOtpSeconds*1000L);
            }

            boolean fromOAuth = "true".equals(req.getParameter("oauth"));

            LoginInfo loginInfo =
                this.loginInfoCache.createLoginInfo(sessionId,this.loginMechanism,
                                                    mfaState,
                                                    locale,tz,
                                                    this.sessionTimeout * 60000L,
                                                    fromOAuth);

            log.info("Successful login of user [{}] with locale [{}] and timezone [{}].",
                    new Object[]{principal.getName(),locale,tz == null ? null : tz.getID()});

            if (groups != null) {

                log.info("Login groups of user [{}] are [{}].",
                        principal.getName(),groups);
            }

            if (!loginInfo.getSessionId().equals(sessionId)) {

                StringBuffer cookie = new StringBuffer();

                //
                // from http://tools.ietf.org/html/rfc6265
                //
                // SID=31d4d96e407aad42; Path=/; Secure; HttpOnly

                cookie.append(this.sessionCookie);
                cookie.append("=");
                cookie.append(loginInfo.getSessionId());

                cookie.append("; Path=/; ");

                if (this.sameSitePolicy != null) {
                    cookie.append("SameSite=");
                    cookie.append(this.sameSitePolicy);
                    cookie.append("; ");
                }

                if (this.secureCookie) {
                    cookie.append("Secure; ");
                }

                cookie.append("HttpOnly");

                resp.setHeader("Set-Cookie",cookie.toString());
            }
            return mfaState;
        }
    }

    public boolean mayReveiceEphemeralOtp(MFAState state) {

        return this.getMfaService().mayReceiveEphemeralOtp(state.getPrincipal(),state.getMfaPrincipal());
    }

    public DomainPasswordLoginService getDomainPasswordLoginService() {
        return this.domainPasswordLoginService;
    }

    public void setDomainPasswordLoginService(
            DomainPasswordLoginService domainPasswordLoginService) {
        this.domainPasswordLoginService = domainPasswordLoginService;
    }

    public LoginInfoCache getLoginInfoCache() {
        return this.loginInfoCache;
    }

    public void setLoginInfoCache(LoginInfoCache loginInfoCache) {
        this.loginInfoCache = loginInfoCache;
    }

    public synchronized void setMfaService(MFAService mfaService) {
        this.mfaService = mfaService;
    }

    public synchronized MFAService getMfaService() {
        return this.mfaService;
    }

    public String getLoginMechanism() {
        return this.loginMechanism;
    }

    public void setLoginMechanism(String loginMechanism) {
        this.loginMechanism = loginMechanism;

        this.loginUrl = "/http-login/" + this.loginMechanism + "/login";
    }

    /**
     * @return Whether calls to {@link #logout(HttpServletRequest)} will logout
     *         all authenticated login mechanisms and not only out own mechnism.
     *         The default value is <code>true</code>.
     */
    public synchronized boolean isLogoutAllMechanisms() {
        return this.logoutAllMechanisms;
    }

    /**
     * @param logoutAllMechanisms Whether calls to {@link #logout(HttpServletRequest)}
     *         will logout all authenticated login mechanisms.
     */
    public synchronized void setLogoutAllMechanisms(boolean logoutAllMechanisms) {
        this.logoutAllMechanisms = logoutAllMechanisms;
    }

    public static Logger getLog() {
        return log;
    }

    public synchronized void setSessionTimeout(int sessionTimeout) {
        this.sessionTimeout = sessionTimeout;
    }

    public synchronized int getSessionTimeout() {
        return this.sessionTimeout;
    }

    public synchronized long getFailureTimeout() {
        return this.failureTimeout;
    }

    public synchronized void setFailureTimeout(long failureTimeout) {
        this.failureTimeout = failureTimeout;
    }

    public synchronized boolean isDoTimeZoneDetection() {
        return this.doTimeZoneDetection;
    }

    public synchronized void setDoTimeZoneDetection(boolean doTimeZoneDetection) {
        this.doTimeZoneDetection = doTimeZoneDetection;
    }

    public synchronized boolean isDoGroupsCheck() {
        return this.doGroupsCheck;
    }

    public synchronized void setDoGroupsCheck(boolean doGroupsCheck) {
        this.doGroupsCheck = doGroupsCheck;
    }

    public synchronized void setSessionCookie(String sessionCookie) {
        this.sessionCookie = sessionCookie;
    }

    public synchronized void setSecureCookie(boolean secureCookie) {
        this.secureCookie = secureCookie;
    }

    public synchronized void setSameSitePolicy(SameSitePolicy sameSitePolicy) {
        this.sameSitePolicy = sameSitePolicy;
    }

    public synchronized void setEphemeralOtpSeconds(int ephemeralOtpSeconds) {
        this.ephemeralOtpSeconds = ephemeralOtpSeconds;
    }

}
