/***********************************************************
 * $Id$
 * 
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 23.08.2018
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.adapter.http;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

import org.clazzes.fancymail.server.api.EMailAddressDTO;
import org.clazzes.fancymail.server.api.EMailDTO;
import org.clazzes.fancymail.server.api.EMailSenderDTO;
import org.clazzes.fancymail.server.api.FancyMailServerSMSService;
import org.clazzes.fancymail.server.api.FancyMailServerService;
import org.clazzes.fancymail.server.api.SMSDTO;
import org.clazzes.fancymail.server.api.SMSDestinationDTO;
import org.clazzes.fancymail.server.api.SMSSenderDTO;
import org.clazzes.login.adapter.http.i18n.Messages;
import org.clazzes.util.sec.DomainPrincipal;
import org.clazzes.util.sec.HashTools;
import org.clazzes.util.sec.MFAPrincipal;
import org.clazzes.util.sec.TokenOtpChecker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A class encapsulating the multiple factor authentication services like
 * checking token OTPs or generating ephemeral OTPs, which will be sent via
 * SMS or e-mail.
 */
public class MFAServiceImpl implements MFAService {

    private static final Logger log = LoggerFactory.getLogger(MFAServiceImpl.class);
    
    private FancyMailServerSMSService smsService;
    private String smsSender;
    
    private FancyMailServerService mailService;
    private String mailSender;

    private TokenOtpChecker tokenOtpChecker;
    
    public void tokenOtpCheckerBound(TokenOtpChecker tokenOtpChecker) {
        
        log.info("Service of type TokenOtpChecker became available, token OTP checks are now activated.");

        synchronized(this) {
            this.tokenOtpChecker = tokenOtpChecker;
        }
    }
    
    public void tokenOtpCheckerUnbound(TokenOtpChecker tokenOtpChecker) {

        log.info("Service of type TokenOtpChecker disappeared, token OTP checks are now disabled.");

        synchronized(this) {
            this.tokenOtpChecker = null;
        }
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.login.adapter.http.MFAService#checkTokenOtp(java.lang.String, java.lang.String[])
     */
    @Override
    public boolean checkTokenOtp(String otp, String[] knownIds) {
        
        TokenOtpChecker checker;
        
        synchronized(this) {
            checker = this.tokenOtpChecker;
        }
        
        try {
            return checker.checkOTP(otp, knownIds);
        } catch (IOException e) {
            log.error("Checking a token OTP failed",e);
            return false;
        }
    }

    public void mailServiceBound(FancyMailServerService mailService) {
        
        if (this.mailSender != null && this.mailSender.length() > 0) {
            
            log.info("Service of type FancyMailServerService became available, mail sending is now activated for sender [{}].",this.mailSender);
            
            EMailSenderDTO sender = mailService.getSenderAddress(this.mailSender);
            
            if (sender == null) {
                
                log.info("Creating new mail sender for address [{}]",this.mailSender);
                
                sender = new EMailSenderDTO();
                
                sender.setPersonalName("clazzes.org Login Service");
                sender.setAddress(this.mailSender);
                
                mailService.insertSender(sender);
            }
        }
        else {
            log.info("Service of type FancyMailServerService became available, but no mail sender is configured, e-mail sending will not be activated.");
            return;
        }

        synchronized (this) {
             
            this.mailService = mailService;
        }
    }
    
    public synchronized void mailServiceUnbound(FancyMailServerService smsService) {
        log.info("Service of type FancyMailServerService disappeared, mail sending is now disabled.");
        this.mailService = null;
    }

    public void smsServiceBound(FancyMailServerSMSService smsService) {
        
        if (this.smsSender != null && this.smsSender.length() > 0) {
            
            log.info("Service of type FancyMailServerSMSService became available, SMS sending is now activated for sender [{}].",this.smsSender);
            
            SMSSenderDTO sender = smsService.getSenderDto(this.smsSender);
            
            if (sender == null) {
                
                log.info("Creating new SMS sender for source number [{}]",this.smsSender);
                
                sender = new SMSSenderDTO();
                
                sender.setPersonalName("msmasml");
                sender.setSourceNumber(this.smsSender);
                
                smsService.insertSender(sender);
            }
        }
        else {
            log.info("Service of type FancyMailServerSMSService became available, but no SMS sender is configured, SMS sending will not be activated.");
            return;
        }
        
        synchronized (this) {
             
            this.smsService = smsService;
        }
    }
    
    public synchronized void smsServiceUnbound(FancyMailServerSMSService smsService) {
        log.info("Service of type FancyMailServerSMSService disappeared, SMS sending is now disabled.");
        this.smsService = null;
    }

    /* (non-Javadoc)
     * @see org.clazzes.login.adapter.http.MFAService#generateEmphemeralOtp(java.util.Locale, java.lang.String, org.clazzes.util.sec.DomainPrincipal, org.clazzes.util.sec.MFAPrincipal)
     */
    @Override
    public String generateEmphemeralOtp(Locale locale, String host, DomainPrincipal principal, MFAPrincipal mfaPrincipal) {
        
        ResourceBundle i18n = Messages.getLocalizedVersion(locale);
        
        FancyMailServerSMSService smsService;
        FancyMailServerService mailService;
        
        synchronized (this) {
            
            smsService = this.smsService;
            mailService = this.mailService;
        }
        
        if (smsService == null && mailService == null) {
            
            log.warn("No service of type FancyMailServerSMSService FancyMailServerService is available, ephemeral OTPs are unavailable.");
            return null;
        }
        
        String otp = HashTools.randomSaltSms(8);
        
        String mobile = mfaPrincipal.getMobileNumber();
        String email = principal.getEMailAddress();
 
        if (smsService != null && mobile != null) {
        
            SMSDTO sms = new SMSDTO();
            
            List<SMSDestinationDTO> destinations= new ArrayList<SMSDestinationDTO>();
            
            SMSDestinationDTO dest = new SMSDestinationDTO();
            dest.setDestNumber(mobile);
            dest.setPersonalName(principal.getPrettyName());
            
            destinations.add(dest);
            
            sms.setSender(this.smsSender);
            sms.setDestinations(destinations);
            
            sms.setText(String.format(locale,i18n.getString("emphemeralOtpSms.body"),principal.getName(),host,otp));
            
            smsService.queueSMS(sms);
        }
        else if (mailService != null && email != null){
            
            EMailDTO mail = new EMailDTO();
            
            EMailAddressDTO dest = new EMailAddressDTO();
            
            dest.setAddress(email);
            dest.setPersonalName(principal.getPrettyName());
            
            List<EMailAddressDTO> to = new ArrayList<EMailAddressDTO>();
            to.add(dest);
            
            mail.setSender(this.mailSender);
            mail.setTo(to);
            
            mail.setSubject(String.format(locale,i18n.getString("emphemeralOtpSms.subject"),principal.getName(),host));
            mail.setBody(String.format(locale,i18n.getString("emphemeralOtpSms.body"),principal.getName(),host,otp));
            
            mailService.queueMail(mail);
        }
        else {
            
            if (smsService != null) {
                
                if (mailService != null) {
                    log.warn("No mobile number or email address for user [{}] to deliver OTP to.",principal.getName());
                }
                else {
                    log.warn("No mobile number for user [{}] to deliver OTP to.",principal.getName()); 
                }
            }
            else {
                log.warn("No email address for user [{}] to deliver OTP to.",principal.getName());
            }
            
            return null;
        }
        
        return otp;
    }

    @Override
    public boolean mayReceiveEphemeralOtp(DomainPrincipal principal,
            MFAPrincipal mfaPrincipal) {
        
        FancyMailServerSMSService smsService;
        FancyMailServerService mailService;
        
        synchronized (this) {
            
            smsService = this.smsService;
            mailService = this.mailService;
        }
        
        String mobile = mfaPrincipal.getMobileNumber();
 
        if (smsService != null && mobile != null) {
            return true;
        }

        String email = principal.getEMailAddress();

        return mailService != null && email != null;
    }

    public void setSmsSender(String smsSender) {
        this.smsSender = smsSender;
    }

    public void setMailSender(String mailSender) {
        this.mailSender = mailSender;
    }
 
    
}
