/***********************************************************
 * $Id$
 * 
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 23.08.2018
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.adapter.http;

import java.security.Principal;
import java.util.List;

import org.clazzes.util.sec.DomainPrincipal;
import org.clazzes.util.sec.MFAPrincipal;

/**
 * A per-mechanism state possibly reflecting Multiple Factor Authentication State.
 */
public class MFAState {

    enum State {
        /**
         * This session is fully authenticated, either because the given
         * principal is not an {@link MFAPrincipal} or a two-factor authentication
         * has successfully been performed.
         */
        AUTHENTICATED,
        /**
         * The MFAPrincipal has a list of known token IDs and this session is awaiting
         * a token authentication.
         */
        TOKEN_PENDING,
        /**
         * The MFAPrincipal has no known token IDs and an ephemeral OTP must be generated
         * to proceed with the two factor authentication.
         */
        EPHEMERAL_NEEDED,
        /**
         * An ephemeral OTP has been generated and this session is waiting the confirmation
         * of the ephemeral OTP.
         */
        EPHEMERAL_PENDING
    }
    
    private final DomainPrincipal principal;
    private final List<? extends Principal> groups;
    private final MFAPrincipal mfaPrincipal;
    private final State state;
    private final String ephemeralOtp;
    private final long ephemeralOtpExpiry;

    /**
     * Generate an initial state after authenticating with username/password.
     * 
     * @param principal The principal, which may or may not implement {@link MFAPrincipal}.
     */
    public MFAState(DomainPrincipal principal, List<? extends Principal> groups) {
        super();
        
        this.principal = principal;
        this.groups = groups;
        
        if (principal  instanceof MFAPrincipal) {
            this.mfaPrincipal = (MFAPrincipal)principal;
            this.state =
                    this.mfaPrincipal.getKnownTokenIds() == null ?
                            State.EPHEMERAL_NEEDED : State.TOKEN_PENDING;
        }
        else {
            this.mfaPrincipal = null;
            this.state = State.AUTHENTICATED;
        }
        this.ephemeralOtp = null;
        this.ephemeralOtpExpiry = 0L;
    }
    
    /**
     * Generate an initial state after authenticating with username/password together with
     * an ephemeral OTP.
     * 
     * @param principal The principal, which may or may not implement {@link MFAPrincipal}.
     * @param ephemeralOtp An ephemeral OTP, which is initially generated, when there is no known
     *                     token ID for a principal.
     * @param ephemeralOtpExpiry The expiry timestamp for the ephemeral OTP.
     */
    public MFAState(DomainPrincipal principal, List<? extends Principal> groups, String ephemeralOtp, long ephemeralOtpExpiry) {
        
        super();
        
        this.principal = principal;
        this.groups = groups;
        
        if (principal  instanceof MFAPrincipal) {
            this.mfaPrincipal = (MFAPrincipal)principal;
            this.state = ephemeralOtp == null ?
                            State.EPHEMERAL_NEEDED : State.EPHEMERAL_PENDING;
        }
        else {
            this.mfaPrincipal = null;
            this.state = State.AUTHENTICATED;
        }

        this.ephemeralOtp = ephemeralOtp;
        this.ephemeralOtpExpiry = ephemeralOtpExpiry;
    }
    
    /**
     * Return a new state, which is augmented to the authenticated state.
     * 
     * @param old
     */
    public MFAState(MFAState old) {
        
        this.principal = old.getPrincipal();
        this.groups = old.getGroups();
        this.mfaPrincipal = old.getMfaPrincipal();
        this.state = State.AUTHENTICATED;
        this.ephemeralOtp = null;
        this.ephemeralOtpExpiry = 0L;
    }
    
    public DomainPrincipal getPrincipal() {
        return this.principal;
    }

    /**
     * @return the groups of this user.
     */
    public List<? extends Principal> getGroups() {
        return this.groups;
    }

    /**
     * @return The MFA principal aspect of the presented principal or
     *         <code>null</code>, if multiple factor authentication is not in effect.
     */
    public MFAPrincipal getMfaPrincipal() {
        return this.mfaPrincipal;
    }

    public State getState() {
        return this.state;
    }

    public long getEphemeralOtpExpiry() {
        return this.ephemeralOtpExpiry;
    }

    public String getEphemeralOtp() {
        return this.ephemeralOtp;
    }
    
}
