/* **********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/
package org.clazzes.login.adapter.http;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.TimeZone;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.clazzes.login.adapter.http.MFAState.State;
import org.clazzes.login.adapter.http.i18n.Messages;
import org.clazzes.util.http.LocaleHelper;
import org.clazzes.util.http.RequestHelper;
import org.clazzes.util.http.UrlHelper;
import org.clazzes.util.http.sec.HttpCheckLoginHelper;
import org.clazzes.util.http.sec.HttpLoginService;
import org.clazzes.util.http.sec.PageTokenService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>The login servlet, which actually provides for the login form according
 * to the documentation for {@link HttpLoginService#getLoginUrl()}.</p>
 *
 * <p>When accepting a GET request, the login form is simply rendered based
 * on the login status of the current HTTP session.</p>
 *
 * <p>When accepting POST requests, the <code>user</code> and <code>password</code>
 * request attributes are inspected and presented to the login service. Based
 * on the result, the login form is rendered.</p>
 */
public class DomainLoginServlet extends HttpServlet {

    private static final long serialVersionUID = 6376913713678650071L;

    private static final Logger log = LoggerFactory.getLogger(DomainLoginServlet.class);

    private static final String XHTML_NS_URI = "http://www.w3.org/1999/xhtml";
    private static final XMLOutputFactory xmlOutputFactory;

    private static final int SC_EXPECT_TOKEN_OTP = 491;
    private static final int SC_EXPECT_EPHEMERAL_OTP = 492;

    private DomainHttpLoginService loginService;
    private PageTokenService pageTokenService;
    private String i18nPrefix;

    static {
        xmlOutputFactory = XMLOutputFactory.newFactory();
        xmlOutputFactory.setProperty(XMLOutputFactory.IS_REPAIRING_NAMESPACES,true);
    }

    private final String getStringWithPfx(ResourceBundle i18n, String key) {

        if (this.i18nPrefix != null) {

            String altKey = this.i18nPrefix + "." + key;

            if (i18n.containsKey(altKey)) {
                return i18n.getString(altKey);
            }
        }

        return i18n.getString(key);
    }

    /**
     * Actually render the login form according to {@link HttpLoginService#getLoginUrl()}.
     *
     * @param locale The user's locale.
     * @param uri The URI where to post the login form data.
     * @param resp The HTTP resonse to write to.
     * @param css The CSS stylesheet to include. If <code>null</code>, the built-in
     *             stylesheet <code>http-login.css</code> will be used.
     * @param doTimeZoneDetection Whether to use time zone detection.
     * @param user The user to write to the login result form.
     * @param status The login status to write to the login result form.
     * @throws IOException Upon I/O errors.
     * @throws ServletException Upon errors related to the servlet API.
     */
    protected void writeLoginForm(Locale locale,
            String uri,
            HttpServletResponse resp,
            String css,
            boolean doTimeZoneDetection,
            String user,
            String pageToken,
            int status,
            String detailMessage,
            boolean allowSms,boolean fromOAuth) throws  IOException, ServletException {

        ResourceBundle i18n = Messages.getLocalizedVersion(locale);

        try {
            String lang = LocaleHelper.toXsLanguage(i18n.getLocale());

            resp.setHeader("X-Frame-Options","SAMEORIGIN");
            resp.setHeader("Content-Language",lang);
            resp.setHeader("Cache-Control","no-cache");
            resp.setHeader("Pragma","no-cache");
            resp.setHeader("Expires","0");
            resp.setContentType("text/html; charset=utf-8");

            resp.getOutputStream().write("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n".getBytes("UTF-8"));

            XMLStreamWriter xsw = xmlOutputFactory.createXMLStreamWriter(resp.getOutputStream(),"UTF-8");

            xsw.setDefaultNamespace(XHTML_NS_URI);

            //xsw.writeStartDocument();

            boolean allowLogout = status == SC_EXPECT_TOKEN_OTP || status == SC_EXPECT_EPHEMERAL_OTP;

            xsw.writeStartElement("html");
            xsw.writeDefaultNamespace(XHTML_NS_URI);
            xsw.writeAttribute("lang",lang);
            xsw.writeAttribute("xml:lang",lang);
            xsw.writeStartElement("head");

            xsw.writeEmptyElement("meta");
            xsw.writeAttribute("http-equiv","Content-Type");
            xsw.writeAttribute("content","text/html; charset=utf-8");

            xsw.writeEmptyElement("link");
            xsw.writeAttribute("type","text/css");
            xsw.writeAttribute("rel","stylesheet");
            xsw.writeAttribute("href",css == null ? "../http-login.css" : css);

            if (doTimeZoneDetection && status != HttpServletResponse.SC_OK) {
                xsw.writeStartElement("script");
                xsw.writeAttribute("type","text/javascript");
                xsw.writeAttribute("charset","utf-8");
                xsw.writeAttribute("src","../jstz.js");
                xsw.writeEndElement();

                xsw.writeStartElement("script");
                xsw.writeCharacters("\nfunction doTimeZoneDetection() {\n");
                xsw.writeCharacters("  if (location.search.match(/tz=(GMT[+\\-][0-9][0-9]:[0-9][0-9]|[\\w\\/]+)/)) {\n");
                xsw.writeCharacters("     document.forms.httpLoginForm.timeZone.value=RegExp.$1;\n");
                xsw.writeCharacters("  } else {\n");
                xsw.writeCharacters("    var tz = jstz.determine();\n");
                xsw.writeCharacters("    document.forms.httpLoginForm.timeZone.value=tz.name();\n");
                xsw.writeCharacters("  }\n");
                xsw.writeCharacters("}\n");
                xsw.writeEndElement();
            }

            xsw.writeStartElement("title");
            xsw.writeCharacters("HTTP Single-Sign-On");
            xsw.writeEndElement(); // </title>

            xsw.writeEndElement(); // </head>

            xsw.writeStartElement("body");

            if (doTimeZoneDetection && status != HttpServletResponse.SC_OK) {
                xsw.writeAttribute("onload","doTimeZoneDetection()");
            }

            // hidden form
            xsw.writeStartElement("form");
            xsw.writeAttribute("id","loginResultForm");

            xsw.writeEmptyElement("input");
            xsw.writeAttribute("type","hidden");
            xsw.writeAttribute("name","status");
            xsw.writeAttribute("value",String.valueOf(status));

            xsw.writeEmptyElement("input");
            xsw.writeAttribute("type","hidden");
            xsw.writeAttribute("name","principal");
            xsw.writeAttribute("value",user);

            xsw.writeEmptyElement("input");
            xsw.writeAttribute("type","hidden");
            xsw.writeAttribute("name",HttpCheckLoginHelper.PAGE_TOKEN_VARIABLE);
            xsw.writeAttribute("value",pageToken);

            xsw.writeEndElement(); // </form>

            // login form
            xsw.writeStartElement("form");
            xsw.writeAttribute("name","httpLoginForm");
            xsw.writeAttribute("action",uri);
            xsw.writeAttribute("method","post");

            xsw.writeEmptyElement("input");
            xsw.writeAttribute("type","hidden");
            xsw.writeAttribute("name",HttpCheckLoginHelper.PAGE_TOKEN_VARIABLE);
            xsw.writeAttribute("value",pageToken);

            if (allowLogout || status == HttpServletResponse.SC_OK) {
                xsw.writeEmptyElement("input");
                xsw.writeAttribute("type","hidden");
                xsw.writeAttribute("name","logout");
                xsw.writeAttribute("value",status == HttpServletResponse.SC_OK ? "true" : "false");
            }

            if (doTimeZoneDetection && status != HttpServletResponse.SC_OK) {
                xsw.writeEmptyElement("input");
                xsw.writeAttribute("type","hidden");
                xsw.writeAttribute("name","timeZone");
                xsw.writeAttribute("value","");
            }

            if (allowSms) {
                xsw.writeEmptyElement("input");
                xsw.writeAttribute("type","hidden");
                xsw.writeAttribute("name","generateSmsOtp");
                xsw.writeAttribute("value","false");
            }

            if (fromOAuth) {
                xsw.writeEmptyElement("input");
                xsw.writeAttribute("type","hidden");
                xsw.writeAttribute("name","oauth");
                xsw.writeAttribute("value","true");
            }

            xsw.writeStartElement("table");
            xsw.writeAttribute("class","http-LoginForm");

            xsw.writeStartElement("tr");
            xsw.writeStartElement("td");
            xsw.writeAttribute("colspan","2");
            xsw.writeAttribute("id","messageTd");

            if (status == HttpServletResponse.SC_FORBIDDEN) {
                if (detailMessage == null) {
                    xsw.writeCharacters(this.getStringWithPfx(i18n, "user-or-password-invalid"));
                } else {
                    xsw.writeCharacters(this.getStringWithPfx(i18n, detailMessage));
                }
            } else if (status == HttpServletResponse.SC_NOT_ACCEPTABLE) {
                xsw.writeCharacters(this.getStringWithPfx(i18n,"too-many-retries"));
            } else if (status == HttpServletResponse.SC_OK) {
                xsw.writeCharacters(this.getStringWithPfx(i18n,"login-ok"));
            } else if (status == SC_EXPECT_TOKEN_OTP) {
                xsw.writeCharacters(this.getStringWithPfx(i18n,"enter-token-otp"));
            } else if (status == SC_EXPECT_EPHEMERAL_OTP) {
                xsw.writeCharacters(this.getStringWithPfx(i18n,"enter-ephemeral-otp"));
            } else {
                xsw.writeCharacters(this.getStringWithPfx(i18n,"enter-user-and-password"));
            }

            xsw.writeEndElement(); // </td>
            xsw.writeEndElement(); // </tr>

            if (status != HttpServletResponse.SC_OK) {

                if (status == SC_EXPECT_TOKEN_OTP) {

                    // Token OTP field
                    xsw.writeStartElement("tr");
                    xsw.writeStartElement("td");
                    xsw.writeStartElement("label");
                    xsw.writeAttribute("for","tokenOtp");
                    xsw.writeCharacters(this.getStringWithPfx(i18n,"token-otp"));
                    xsw.writeEndElement(); // </label>
                    xsw.writeEndElement(); // </td>
                    xsw.writeStartElement("td");
                    xsw.writeEmptyElement("input");
                    xsw.writeAttribute("class","http-login-TokenOtpTextBox");
                    xsw.writeAttribute("title",this.getStringWithPfx(i18n,"enter-token-otp"));
                    xsw.writeAttribute("type","text");
                    xsw.writeAttribute("name","tokenOtp");
                    xsw.writeAttribute("id","tokenOtp");
                    xsw.writeAttribute("autocomplete","off");
                    xsw.writeAttribute("autofocus", "autofocus");
                    xsw.writeEndElement(); // </td>
                    xsw.writeEndElement(); // </tr>
                }
                else if (status == SC_EXPECT_EPHEMERAL_OTP) {

                    // Token OTP field
                    xsw.writeStartElement("tr");
                    xsw.writeStartElement("td");
                    xsw.writeStartElement("label");
                    xsw.writeAttribute("for","ephemeralOtp");
                    xsw.writeCharacters(this.getStringWithPfx(i18n,"ephemeral-otp"));
                    xsw.writeEndElement(); // </label>
                    xsw.writeEndElement(); // </td>
                    xsw.writeStartElement("td");
                    xsw.writeEmptyElement("input");
                    xsw.writeAttribute("class","http-login-TextBox");
                    xsw.writeAttribute("title",this.getStringWithPfx(i18n,"enter-ephemeral-otp"));
                    xsw.writeAttribute("type","text");
                    xsw.writeAttribute("name","ephemeralOtp");
                    xsw.writeAttribute("id","ephemeralOtp");
                    xsw.writeAttribute("autocomplete","off");
                    xsw.writeAttribute("autofocus", "autofocus");
                    xsw.writeEndElement(); // </td>
                    xsw.writeEndElement(); // </tr>
                }
                else {
                    // User field
                    xsw.writeStartElement("tr");
                    xsw.writeStartElement("td");
                    xsw.writeStartElement("label");
                    xsw.writeAttribute("for","user");
                    xsw.writeCharacters(this.getStringWithPfx(i18n,"user"));
                    xsw.writeEndElement(); // </label>
                    xsw.writeEndElement(); // </td>
                    xsw.writeStartElement("td");
                    xsw.writeEmptyElement("input");
                    xsw.writeAttribute("class","http-login-TextBox");
                    xsw.writeAttribute("title",
                            String.format(locale,this.getStringWithPfx(i18n,"domainUserTitle"),
                                    this.loginService.getDefaultDomain())); // FUTURE: Domain list based on this.loginService.getDomains()?
                    xsw.writeAttribute("placeholder",this.getStringWithPfx(i18n,"domainUserPlaceholder"));
                    xsw.writeAttribute("type","text");
                    xsw.writeAttribute("name","user");
                    xsw.writeAttribute("id","user");
                    xsw.writeAttribute("autofocus", "autofocus");
                    xsw.writeEndElement(); // </td>
                    xsw.writeEndElement(); // </tr>

                    // Password
                    xsw.writeStartElement("tr");
                    xsw.writeStartElement("td");
                    xsw.writeStartElement("label");
                    xsw.writeAttribute("for","password");
                    xsw.writeCharacters(this.getStringWithPfx(i18n,"password"));
                    xsw.writeEndElement(); // </label>
                    xsw.writeEndElement(); // </td>
                    xsw.writeStartElement("td");
                    xsw.writeEmptyElement("input");
                    xsw.writeAttribute("class","http-login-PasswordTextBox");
                    xsw.writeAttribute("placeholder",this.getStringWithPfx(i18n,"password"));
                    xsw.writeAttribute("type","password");
                    xsw.writeAttribute("name","password");
                    xsw.writeAttribute("id","password");
                    xsw.writeEndElement(); // </td>
                    xsw.writeEndElement(); // </tr>
                }

                if (fromOAuth) {

                    String href = "/oauth-login/login";

                    href = UrlHelper.appendQueryParameterToUrl(href,"locale",lang);

                    xsw.writeStartElement("tr");
                    xsw.writeStartElement("td");
                    xsw.writeAttribute("colspan","2");

                    xsw.writeStartElement("a");
                    xsw.writeAttribute("href",href);

                    xsw.writeCharacters(i18n.getString("backToOAuth"));
                    xsw.writeEndElement(); // </a>

                    xsw.writeEndElement(); // </td>
                    xsw.writeEndElement(); // </tr>
                }
            }

            xsw.writeStartElement("tr");
            xsw.writeStartElement("td");
            xsw.writeAttribute("colspan","2");

            xsw.writeStartElement("fieldset");
            xsw.writeAttribute("id","httpLoginFields");

            xsw.writeEmptyElement("input");
            xsw.writeAttribute("class","http-login-Button");
            xsw.writeAttribute("type","submit");

            String submitMsg = status == HttpServletResponse.SC_OK ?
                    this.getStringWithPfx(i18n,"loggingOut") :
                        this.getStringWithPfx(i18n,"checkingCredentials");

           xsw.writeAttribute("onClick",
                            "document.getElementById(\"httpLoginFields\").disabled = true; document.getElementById(\"messageTd\").innerHTML=\""+
                                    submitMsg+
                            "\"; document.forms.httpLoginForm.submit(); return false;");

           xsw.writeAttribute("value",status == HttpServletResponse.SC_OK ?
                   this.getStringWithPfx(i18n,"do-logout") :
                       this.getStringWithPfx(i18n,"do-login")   );

           if (allowSms) {

               xsw.writeCharacters("\u00a0");
               xsw.writeEmptyElement("input");
               xsw.writeAttribute("class","http-login-Button");
               xsw.writeAttribute("type","button");

               String sendMsg = this.getStringWithPfx(i18n,"generatingEphemeralOtp");

               xsw.writeAttribute("onClick",
                       "document.getElementById(\"httpLoginFields\").disabled = true;  document.forms.httpLoginForm.generateSmsOtp.value='true'; document.getElementById(\"messageTd\").innerHTML=\""+
                               sendMsg+
                       "\"; document.forms.httpLoginForm.submit(); return false;");

               xsw.writeAttribute("value",this.getStringWithPfx(i18n,"do-generate-otp"));
           }

           if (allowLogout) {

               xsw.writeCharacters("\u00a0");
               xsw.writeStartElement("input");
               xsw.writeAttribute("class","http-login-Button");
               xsw.writeAttribute("type","button");

               xsw.writeAttribute("onClick",
                       "document.getElementById(\"httpLoginFields\").disabled = true;  document.forms.httpLoginForm.logout.value='true'; document.getElementById(\"messageTd\").innerHTML=\""+
                               this.getStringWithPfx(i18n,"loggingOut")+
                       "\"; document.forms.httpLoginForm.submit(); return false;");

               xsw.writeAttribute("value",this.getStringWithPfx(i18n,"do-logout"));
               xsw.writeEndElement(); // </a>
           }

           xsw.writeEndElement(); // </fieldset>

           xsw.writeEndElement(); // </td>
           xsw.writeEndElement(); // </tr>

           xsw.writeEndElement(); // </table>

           xsw.writeEndElement(); // </form>

           xsw.writeEndElement(); // </body>

           xsw.writeEndElement(); // </html>

           xsw.writeEndDocument();
           xsw.close();

           resp.flushBuffer();

        } catch (XMLStreamException e) {

            throw new ServletException("Error setting XML stream writer",e);
        }
    }

    protected static Locale getRequestLocale(HttpServletRequest req) {

        Locale locale = req.getLocale();

        String locale_s = req.getParameter("locale");

        if (locale_s != null) {
            locale = LocaleHelper.localeFromXsLanguage(locale_s);
        }

        return locale;
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException {

        MFAState login = this.loginService.checkMFALogin(req);
        boolean fromOAuth = "true".equals(req.getParameter("oauth"));

        this.writeResponse(req,resp,login,false,getRequestLocale(req),null,fromOAuth);
    }

    /**
     * Write the response for both POST and GET requests. This method
     * determines the login status and handles the optional <code>locale</code>
     * and <code>css</code> GET properties to the request.
     *
     * @param req The request.
     * @param resp The response.
     * @param login The login or <code>null</code>, if no user is logged in.
     * @param wasPost <code>true</code> for POST requests.
     * @throws ServletException
     * @throws IOException
     */
    protected void writeResponse(HttpServletRequest req, HttpServletResponse resp, MFAState login, boolean wasPost, Locale locale, String detailMessage,boolean fromOAuth)
            throws ServletException, IOException {

        String css = req.getParameter("css");

        int status;
        String user;
        String pageToken = null;
        boolean allowSms = false;

        if (login == null) {

            if (!this.loginService.checkPermission(req,this.loginService.getLoginUrl()))
                status = HttpServletResponse.SC_NOT_ACCEPTABLE;
            else if (wasPost)
                status = HttpServletResponse.SC_FORBIDDEN;
            else
                status = HttpServletResponse.SC_UNAUTHORIZED;

            user = "";
        }
        else {

            user = login.getPrincipal().getName();

            if (login.getState() == State.AUTHENTICATED) {

                status = HttpServletResponse.SC_OK;
            }
            else if (login.getState() == State.TOKEN_PENDING) {

                status = SC_EXPECT_TOKEN_OTP;
                allowSms = this.loginService.mayReveiceEphemeralOtp(login);
            }
            else if (login.getState() == State.EPHEMERAL_PENDING) {
                status = SC_EXPECT_EPHEMERAL_OTP;
            }
            else {
                // this should not happen.
                status = HttpServletResponse.SC_FORBIDDEN;
            }

            if (wasPost) {
                pageToken = req.getParameter(HttpCheckLoginHelper.PAGE_TOKEN_VARIABLE);
            }
        }

        if (pageToken == null) {
            pageToken = this.pageTokenService.getPageToken(req);
        }

        // append 'locale' and/or 'css' GET properties to the form action URL.
        String uri = req.getRequestURI();

        String locale_s = req.getParameter("locale");

        if (locale_s != null) {
            uri = UrlHelper.appendQueryParameterToUrl(uri,"locale",locale_s);
        }

        if (css != null) {
            uri = UrlHelper.appendQueryParameterToUrl(uri,"css",css);
        }

        this.writeLoginForm(locale,uri,resp,css,this.loginService.isDoTimeZoneDetection(),user,pageToken,status,detailMessage,allowSms,fromOAuth);
    }

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException {

        MFAState login=null;
        String detailMessage = null;

        String logout = req.getParameter("logout");

        boolean isLogout = "true".equals(logout);
        boolean fromOAuth = "true".equals(req.getParameter("oauth"));

        Locale locale = getRequestLocale(req);

        if (isLogout) {

            fromOAuth |= this.loginService.logoutCheckForOAuth(req);
        }
        else {

            try {

                String pageToken = req.getParameter(HttpCheckLoginHelper.PAGE_TOKEN_VARIABLE);

                if (this.pageTokenService.checkPageToken(req,pageToken)) {

                    String sendSms = req.getParameter("generateSmsOtp");
                    String tokenOtp = req.getParameter("tokenOtp");
                    String ephemeralOtp = req.getParameter("ephemeralOtp");

                    if ("true".equals(sendSms)) {

                        login = this.loginService.generateSmsToken(req,locale);
                    }
                    else if (tokenOtp != null) {
                        login = this.loginService.checkTokenOtp(req,resp,tokenOtp,locale);
                    }
                    else if (ephemeralOtp != null) {

                        login = this.loginService.checkEphemeralOtp(req,resp,ephemeralOtp,locale);
                    }
                    else {

                        String user = req.getParameter("user");
                        String password = req.getParameter("password");

                        int slashIdx = Math.max(user.indexOf('/'), user.indexOf('\\'));

                        String domain;

                        if (slashIdx < 0) {

                            int atIdx = user.indexOf('@');

                            if (atIdx < 0) {
                                // domain/user
                                domain = this.loginService.getDefaultDomain();
                            }
                            else {
                                // user@domain
                                domain = user.substring(atIdx+1);
                                user = user.substring(0,atIdx);
                            }
                        }
                        else {
                            domain = user.substring(0,slashIdx);
                            user = user.substring(slashIdx+1);
                        }

                        String tz_s = req.getParameter("timeZone");

                        TimeZone tz;

                        if (tz_s == null || tz_s.isEmpty()) {

                            tz = null;
                        }
                        else {
                            tz = TimeZone.getTimeZone(tz_s);
                        }

                        login = this.loginService.tryLogin(req,resp,domain,user,password,locale,tz);
                    }
                }
            } catch(RuntimeException e) {
                log.error("Caught exception during login",e);
                this.loginService.logout(req);
                login = null;

                Throwable cause = e.getCause();
                if (cause != null) {
                    if (cause instanceof CertificateExpiredException) {
                        detailMessage = "valid-certificate-expired";
                    } else if (cause instanceof CertificateNotYetValidException) {
                        detailMessage = "no-valid-certificate";
                    }
                }
            } catch (Throwable e) {
                log.error("Caught exception during login",e);
                this.loginService.logout(req);
                login = null;
            }

            if (login == null) {

                log.error("Bad login from [{}].", RequestHelper.getRealRemoteIP(req));

                long failureTimeout = this.loginService.getFailureTimeout();

                if (failureTimeout > 0L) {
                    try {
                        Thread.sleep(failureTimeout);
                    } catch (InterruptedException e) {
                        throw new InterruptedIOException("Wait after HTTP login failure has been interupted.");
                    }
                }
            }
        }
        this.writeResponse(req,resp,login,!isLogout,locale,detailMessage,fromOAuth);
    }

    public void setLoginService(DomainHttpLoginService loginService) {
        this.loginService = loginService;
    }

    public void setPageTokenService(PageTokenService pageTokenService) {
        this.pageTokenService = pageTokenService;
    }

    public void setI18nPrefix(String i18nPrefix) {
        this.i18nPrefix = i18nPrefix;
    }

    @Override
    public String getServletInfo() {

        return DomainLoginServlet.class.getSimpleName();
    }

}
