/***********************************************************
 * $Id$
 * 
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.adapter.http;

import java.util.Dictionary;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.Map.Entry;

import org.clazzes.util.http.osgi.DefaultHttpContext;
import org.clazzes.util.http.sec.HttpLoginService;
import org.clazzes.util.http.sec.PageTokenService;
import org.clazzes.util.osgi.ServiceMapListener2;
import org.clazzes.util.sec.DomainPasswordLoginService;
import org.osgi.framework.Bundle;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.cm.ManagedService;
import org.osgi.service.http.HttpService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A listener for {@link DomainPasswordLoginService} instances, which
 * exports HttpLoginService
 */
public class LoginServiceListener implements ServiceMapListener2, ManagedService {

	private static final Logger log = LoggerFactory.getLogger(LoginServiceListener.class);
	
	private HttpService httpService;
	private Bundle bundle;
	private LoginInfoCache loginInfoCache;
	private PageTokenService pageTokenService;
	private MFAService mfaService;
	private int sessionTimeout;
	private long failureTimeout;
	private String sessionCookie;
	private boolean secureCookie;
	private SameSitePolicy sameSitePolicy;
	private boolean doTimeZoneDetection;
	private boolean doGroupsCheck;
        private boolean logoutAllMechanisms;
	private int ephemeralOtpSeconds;

	private static final class ServiceInfo {
		
		private final ServiceRegistration<HttpLoginService> httpLoginServiceRegistration;
		private final DomainHttpLoginService domainHttpLoginService;
	    private final String i18nPrefix;
		private String servletAlias;
		
		public ServiceInfo(ServiceRegistration<HttpLoginService> httpLoginServiceRegistration,
				DomainHttpLoginService domainHttpLoginService,
				Map<String,Object> serviceProperties) {
			super();
			this.httpLoginServiceRegistration = httpLoginServiceRegistration;
			this.domainHttpLoginService = domainHttpLoginService;
			
			Object i18nPfx = null;
			
			if (serviceProperties != null) {
			   i18nPfx = serviceProperties.get("login.i18n.prefix"); 
			}
			
			this.i18nPrefix = i18nPfx == null ? null : i18nPfx.toString();
		}

		public ServiceRegistration<HttpLoginService> getHttpLoginServiceRegistration() {
			return this.httpLoginServiceRegistration;
		}
		public DomainHttpLoginService getDomainHttpLoginService() {
			return this.domainHttpLoginService;
		}
		public String getI18nPrefix() {
            return this.i18nPrefix;
        }

        public String getServletAlias() {
			return this.servletAlias;
		}

		public void setServletAlias(String servletAlias) {
			this.servletAlias = servletAlias;
		}
	}
	
	private Map<String,ServiceInfo> registrations;
	
	private static final String makeServletAlias(String mechanism) {
		return "/http-login/" + mechanism;
	}
	
	private final void deregisterLoginServlet(ServiceInfo serviceInfo) {
		
		if (serviceInfo.getServletAlias() != null) {
		
			log.info("Deregistering login servlet [{}]",serviceInfo.getServletAlias());
			this.httpService.unregister(serviceInfo.getServletAlias());
			serviceInfo.setServletAlias(null);
		}
	}
	
	private final void registerLoginServlet(String mechanism, ServiceInfo serviceInfo) {
		
		DomainLoginServlet loginServlet = new DomainLoginServlet();
		loginServlet.setLoginService(serviceInfo.getDomainHttpLoginService());
		loginServlet.setPageTokenService(this.pageTokenService);
		loginServlet.setI18nPrefix(serviceInfo.getI18nPrefix());
		
		String alias = makeServletAlias(mechanism);
		log.info("Registering login servlet [{}]",alias);
		try {
			
			this.httpService.registerServlet(alias,loginServlet,null,DefaultHttpContext.getInstance());
			serviceInfo.setServletAlias(alias);
			
		} catch (Exception e) {
			log.error("Error registering login servlet ["+alias+"]",e);
		}
	}
	
	public LoginServiceListener() {
		this.sessionTimeout = 180;
		this.logoutAllMechanisms = true;
	}
	
	public synchronized void httpServiceBound(HttpService httpService) {
	
		this.httpService = httpService;
		
		if (this.registrations != null) {
			
			for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
				this.registerLoginServlet(entry.getKey(),entry.getValue());
			}
		}		
	}
	
	public synchronized void httpServiceUnbound(HttpService httpService) {
		
		if (this.httpService == null) {
			
			log.warn("HttpService became unavailable without prior registration.");
			return;
		}
		
		if (this.registrations != null) {
		
			for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
				this.deregisterLoginServlet(entry.getValue());
			}
		}
		
		this.httpService = null;
	}

	@Override
	public synchronized void serviceBound(String mechanism, Object service, Map<String,Object> serviceProperties) {
		
		DomainPasswordLoginService domainPasswordLoginService = (DomainPasswordLoginService)service;
		
		DomainHttpLoginService loginService = new DomainHttpLoginService();
		loginService.setLoginInfoCache(this.loginInfoCache);
		loginService.setMfaService(this.mfaService);
		loginService.setLoginMechanism(mechanism);
		loginService.setDomainPasswordLoginService(domainPasswordLoginService);
		loginService.setFailureTimeout(this.failureTimeout);
		loginService.setSessionTimeout(this.sessionTimeout);
		loginService.setDoTimeZoneDetection(this.doTimeZoneDetection);
		loginService.setDoGroupsCheck(this.doGroupsCheck);
                loginService.setLogoutAllMechanisms(this.logoutAllMechanisms);
		loginService.setSecureCookie(this.secureCookie);
		loginService.setSameSitePolicy(this.sameSitePolicy);
                loginService.setSessionCookie(this.sessionCookie);
		loginService.setEphemeralOtpSeconds(this.ephemeralOtpSeconds);
		
		log.info("Registering HttpLoginService for mechanism [{}]",mechanism);
		
		Dictionary<String,String> properties = new Hashtable<String,String>();
		properties.put(HttpLoginService.LOGIN_MECHANISM_KEY,mechanism);
		
		@SuppressWarnings("unchecked")
        ServiceRegistration<HttpLoginService> httpLoginServiceRegistration =
				(ServiceRegistration<HttpLoginService>)
				this.bundle.getBundleContext().registerService(HttpLoginService.class.getName(),loginService,properties);
		
		ServiceInfo info = new ServiceInfo(httpLoginServiceRegistration,loginService,serviceProperties);
		
		if (this.httpService != null) {
			this.registerLoginServlet(mechanism,info);
		}

		if (this.registrations == null) {
			this.registrations = new HashMap<String,ServiceInfo>();
		}
		
		this.registrations.put(mechanism,info);
	}

	/* (non-Javadoc)
	 * @see org.clazzes.util.osgi.ServiceMapListener#serviceUnbound(java.lang.String, java.lang.Object)
	 */
	@Override
	public synchronized void serviceUnbound(String mechanism, Object service, Map<String,Object> serviceProperties) {

		ServiceInfo info = this.registrations.remove(mechanism);
		
		if (info == null) {
			
			log.warn("DomainPasswordLoginService for mechanism [{}] became unavailable wihtout prior registration.",mechanism);
			return;
		}
		
		log.info("Unregistering HttpLoginService for mechanism [{}]",mechanism);
		info.getHttpLoginServiceRegistration().unregister();
		
		if (this.httpService != null) {
			this.deregisterLoginServlet(info);
		}
		
		if (this.registrations.isEmpty()) {
			this.registrations = null;
		}
	}

	@SuppressWarnings("rawtypes")
	@Override
	public synchronized void updated(Dictionary properties) throws ConfigurationException {
		
		if (log.isDebugEnabled()) {
			log.debug("updated called: properties=[{}]",properties);
		}
		
		Object sessionTimeout_s = properties.get("sessionTimeout");
		
		if (sessionTimeout_s != null) {
			
			this.sessionTimeout = Integer.parseInt(sessionTimeout_s.toString());
			
			if (this.registrations != null) {
				
				for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
					if (log.isDebugEnabled()) {
						log.debug("Setting sessionTimeout [{}] on login service with mechanism [{}]",
								this.sessionTimeout,entry.getKey());
					}
					entry.getValue().getDomainHttpLoginService().setSessionTimeout(this.sessionTimeout);
				}
			}
		}		

		Object failureTimeout_s = properties.get("failureTimeout");
		
		if (failureTimeout_s != null) {
			
			this.failureTimeout = Integer.parseInt(failureTimeout_s.toString());
			
			if (this.registrations != null) {
				
				for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
					if (log.isDebugEnabled()) {
						log.debug("Setting failureTimeout [{}] on login service with mechanism [{}]",
								this.failureTimeout,entry.getKey());
					}
					entry.getValue().getDomainHttpLoginService().setFailureTimeout(this.failureTimeout);
				}
			}
		}		

                Object doTimeZoneDetection_s = properties.get("doTimeZoneDetection");
                
                if (doTimeZoneDetection_s != null) {
                        
                        this.doTimeZoneDetection = Boolean.parseBoolean(doTimeZoneDetection_s.toString());
                        
                        if (this.registrations != null) {
                                
                                for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
                                        if (log.isDebugEnabled()) {
                                                log.debug("Setting doTimeZoneDetection [{}] on login service with mechanism [{}]",
                                                                this.doTimeZoneDetection,entry.getKey());
                                        }
                                        entry.getValue().getDomainHttpLoginService().setDoTimeZoneDetection(this.doTimeZoneDetection);
                                }
                        }
                }
                
                Object doGroupsCheck_s = properties.get("doGroupsCheck");
                
                if (doGroupsCheck_s != null) {
                        
                        this.doGroupsCheck = Boolean.parseBoolean(doGroupsCheck_s.toString());
                        
                        if (this.registrations != null) {
                                
                                for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
                                        if (log.isDebugEnabled()) {
                                                log.debug("Setting doGroupsCheck [{}] on login service with mechanism [{}]",
                                                                this.doGroupsCheck,entry.getKey());
                                        }
                                        entry.getValue().getDomainHttpLoginService().setDoGroupsCheck(this.doGroupsCheck);
                                }
                        }
                }
                
		Object sessionCookie_s = properties.get("sessionCookie");
		
		if (sessionCookie_s != null) {
			
			this.sessionCookie = sessionCookie_s.toString();
			
			char c = this.sessionCookie.charAt(0);
			
			if ((c<'a' || c>'z') && (c<'A' || c>'Z')) {
				throw new ConfigurationException("sessionCookie","sessionĆookie does not start with an ASCII letter.");
			}
			
			for (int i = 1; i < this.sessionCookie.length(); ++i) {
				
				c = this.sessionCookie.charAt(i);
				
				if ((c<'a' || c>'z') && (c<'A' || c>'Z') && (c<'0' || c>'9') && c!= '_') {
					throw new ConfigurationException("sessionCookie","sessionĆookie contains chracters other than ASCII numbers, letters or underscores.");
				}
			}
			
			if (this.registrations != null) {
				
				for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
					if (log.isDebugEnabled()) {
						log.debug("Setting sessionCookie [{}] on login service with mechanism [{}]",
								this.sessionCookie,entry.getKey());
					}
					entry.getValue().getDomainHttpLoginService().setSessionCookie(this.sessionCookie);
				}
			}
		}
		
		Object secureCookie_s = properties.get("secureCookie");
		
		if (secureCookie_s != null) {
			
			this.secureCookie = Boolean.parseBoolean(secureCookie_s.toString());
			
			if (this.registrations != null) {
				
				for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
					if (log.isDebugEnabled()) {
						log.debug("Setting secureCookie [{}] on login service with mechanism [{}]",
								this.secureCookie,entry.getKey());
					}
					entry.getValue().getDomainHttpLoginService().setSecureCookie(this.secureCookie);
				}
			}
		}

        Object sameSitePolicy_s = properties.get("sameSitePolicy");
                
        if (sameSitePolicy_s != null) {
            
            String v= sameSitePolicy_s.toString().trim();
            
            this.sameSitePolicy = v.isEmpty() ? null : SameSitePolicy.valueOf(v);
            
            if (this.registrations != null) {
                
                for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
                    if (log.isDebugEnabled()) {
                        log.debug("Setting sameSitePolicy [{}] on login service with mechanism [{}]",
                                this.sameSitePolicy,entry.getKey());
                    }
                    entry.getValue().getDomainHttpLoginService().setSameSitePolicy(this.sameSitePolicy);
                }
            }
        }
        
        Object logoutAllMechanisms_s = properties.get("logoutAllMechanisms");
        
        if (logoutAllMechanisms_s != null) {
            
            this.logoutAllMechanisms = Boolean.parseBoolean(logoutAllMechanisms_s.toString());
            
            if (this.registrations != null) {
                
                for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
                    if (log.isDebugEnabled()) {
                        log.debug("Setting logoutAllMechanisms [{}] on login service with mechanism [{}]",
                                this.logoutAllMechanisms,entry.getKey());
                    }
                    entry.getValue().getDomainHttpLoginService().setLogoutAllMechanisms(this.logoutAllMechanisms);
                }
            }
        }

        Object ephemeralOtpSeconds_s = properties.get("ephemeralOtpSeconds");
        
        if (ephemeralOtpSeconds_s != null) {
            
            this.ephemeralOtpSeconds = Integer.parseInt(ephemeralOtpSeconds_s.toString());
            
            if (this.registrations != null) {
                
                for (Entry<String, ServiceInfo> entry : this.registrations.entrySet()) {
                    if (log.isDebugEnabled()) {
                        log.debug("Setting ephemeralOtpSeconds [{}] on login service with mechanism [{}]",
                                this.logoutAllMechanisms,entry.getKey());
                    }
                    entry.getValue().getDomainHttpLoginService().setEphemeralOtpSeconds(this.ephemeralOtpSeconds);
                }
            }
        }
	}

	public void setBundle(Bundle bundle) {
		this.bundle = bundle;
	}

	public void setLoginInfoCache(LoginInfoCache loginInfoCache) {
		this.loginInfoCache = loginInfoCache;
	}

    public void setPageTokenService(PageTokenService pageTokenService) {
        this.pageTokenService = pageTokenService;
    }

    public void setMfaService(MFAService mfaService) {
        this.mfaService = mfaService;
    }

}
