/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/

package org.clazzes.login.htpasswd.jaas;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.Map;

import org.mindrot.jbcrypt.BCrypt;

public abstract class Main {

    private static final String PASSWD_CHARACTERS = "abcdefghijklmnopqrstuvwxyz01234567890";

    public static void usage(PrintStream os) {

        os.println("Usage: org.clazzes.login.htpasswd.jaas.main <cmd> [<parameter> ...]");
        os.println("");
        os.println("  <cmd> is one of:");
        os.println("    adduser <htpasswd> <user> [<password>]");
        os.println("        Add a user <user> to the file <htpasswd>.");
        os.println("        If <password> is not given, generate a random password");
        os.println("        and write ist to the console.");
        os.println("        If <user> exists, its password is changed.");
        os.println("    deluser <htpasswd> <user> [<user> ...]");
        os.println("        Delete one or more users <user> from the file <htpasswd>");
    }

    private static final String genPasswd(int length) {

        SecureRandom rnd = new SecureRandom();

        char[] chars = new char[length];

        for (int i=0;i<length;++i) {
            chars[i] = PASSWD_CHARACTERS.charAt(rnd.nextInt(PASSWD_CHARACTERS.length()));
        }

        return String.valueOf(chars);
    }

    private static final Map<String, String> readHtpasswdFile(Path path) throws IOException {
        try (var reader = Files.newBufferedReader(path, StandardCharsets.UTF_8)) {
            return HtpasswdFileHelper.readHtpasswdFile(reader);
        }
    }

    private static final void writeHtpasswdFile(Path path, Map<String, String> data) throws IOException {
        try (var writer = Files.newBufferedWriter(path, StandardCharsets.UTF_8)) {
            HtpasswdFileHelper.writeHtpasswdFile(writer, data);;
        }
    }

    public static String adduser(String... args) throws IOException {

        int iarg = 1;
        boolean do_wait = false;

        while (iarg < args.length) {

            if ("-wait".equals(args[iarg])) {
                ++iarg;
                do_wait = true;
            }
            else {
                break;
            }
        }

        if (iarg+2 > args.length) {
            System.out.println("Too less arguments for adduser command.");
            usage(System.out);
            System.exit(64);
        }

        if (iarg+3 < args.length) {
            System.out.println("Too many arguments for adduser command.");
            usage(System.out);
            System.exit(64);
        }

        File file = new File(args[iarg++]);
        String user = args[iarg++];

        Map<String,String> users;

        if (file.exists()) {
            users = readHtpasswdFile(file.toPath());
            System.out.println("Successfully read ["+users.size()+
                               "] users from file ["+file+
                               "].");
        }
        else {
            users = new HashMap<String,String>();
            System.out.println("File ["+file+
                               "] does not exist, will create it.");
        }

        String passwd;
        String ret;

        if (iarg < args.length) {
            passwd = args[iarg++];
            ret = null;
        }
        else {
            ret = passwd = genPasswd(12);
            System.out.println("Generated password ["+passwd+
                               "] for user ["+user+
                               "], please store it in a safe place.");

            if (do_wait) {
                System.out.print("Press any key to continue...");
                System.in.read();
                System.out.println("OK");
            }
        }

        String salt = BCrypt.gensalt();

        String encrypted = BCrypt.hashpw(passwd,salt);

        users.put(user,encrypted);

        writeHtpasswdFile(file.toPath(), users);

        System.out.println("Successfully wrote ["+users.size()+
                           "] users to file ["+file+
                           "].");
        return ret;
    }

    public static void deluser(String... args) throws IOException {

        File file = new File(args[1]);

        if (!file.exists()) {
            System.out.println("File ["+file+
                               "] does not exist, no users to delete.");
            return;
        }

        Map<String,String> users = readHtpasswdFile(file.toPath());
        System.out.println("Successfully read ["+users.size()+
                           "] users from file ["+file+
                           "].");

        int ndeleted = 0;

        for (int iarg = 2;iarg<args.length;++iarg) {

            String user = args[iarg];
            if (users.remove(user) == null) {
                System.out.println("User ["+user+"] did not exist.");
            }
            else {
                System.out.println("Deleted user ["+user+"].");
                ++ndeleted;
            }
        }

        if (ndeleted > 0) {
            writeHtpasswdFile(file.toPath(),users);

            System.out.println("Successfully wrote ["+users.size()+
                               "] users to file ["+file+
                               "].");
        }
    }

    public static void main(String[] args) {

        if (args.length < 3) {
            usage(System.out);
            System.exit(64);
        }

        String cmd = args[0];

        try {
            switch(cmd) {
                case "adduser":
                adduser(args);
                break;

                case "deluser":
                deluser(args);
                break;

                default:
                System.out.println("Invalid command ["+cmd+"].");
                usage(System.out);
                System.exit(64);
            }
        }
        catch(Exception e) {
            e.printStackTrace(System.err);
            System.exit(1);
        }
    }
}
