/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/

package org.clazzes.login.htpasswd.jaas;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class HtpasswdFileHelper {

    /**
     * Read an htpasswd file.
     *
     * @param file The colon-separated file to read.
     * @return A map from the username to the hashed password.
     * @throws IOException
     */
    public static Map<String,String> readHtpasswdFile(Reader file) throws IOException {

        Pattern entry = Pattern.compile("^([^:]+):(.+)");

        Map<String,String> htUsers = new HashMap<String,String>();

        try (Scanner scanner = new Scanner(file)) {

            while (scanner.hasNextLine()) {
                String line = scanner.nextLine().trim();
                if (!line.isEmpty() &&  !line.startsWith("#")) {
                    Matcher m = entry.matcher(line);
                    if (m.matches()) {
                        htUsers.put(m.group(1), m.group(2));
                    }
                }
            }
        }
        return htUsers;
    }

    public static void writeHtpasswdFile(Writer file, Map<String,String> htUsers) throws IOException {

        try (Writer w = file) {

            List<String> users = new ArrayList<String>(htUsers.size());

            users.addAll(htUsers.keySet());
            Collections.sort(users);

            Pattern valid = Pattern.compile("[^\\p{Space}\\p{Cntrl}:]+");

            for (String user: users) {

                String hash = htUsers.get(user);

                if (!valid.matcher(user).matches()) {
                    throw new IllegalArgumentException("User ["+user+"] contains invalid characters");
                }

                if (!valid.matcher(hash).matches()) {
                    throw new IllegalArgumentException("Hash for user ["+user+"] contains invalid characters");
                }

                w.write(user);
                w.write(":");
                w.write(hash);
                w.write("\n");
            }
        }
    }
}
