/*
* Licensed to the Apache Software Foundation (ASF) under one
* or more contributor license agreements.  See the NOTICE file
* distributed with this work for additional information
* regarding copyright ownership.  The ASF licenses this file
* to you under the Apache License, Version 2.0 (the
* "License"); you may not use this file except in compliance
* with the License.  You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an
* "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.  See the License for the
* specific language governing permissions and limitations
* under the License.
*/

package org.clazzes.login.htpasswd.jaas;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Consumer;
import java.util.regex.Pattern;

import javax.security.auth.login.LoginException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HtpasswdAuthServiceFactory {
    private HtpasswdAuthServiceFactoryConfiguration config;
    public void setConfig(HtpasswdAuthServiceFactoryConfiguration config) {
        Objects.requireNonNull(config);
        this.config = config;
    }

    public HtpasswdAuthServiceFactory(
            HtpasswdAuthServiceFactoryConfiguration config) {
        Objects.requireNonNull(config);
        this.config = config;
    }

    private static final Logger log = LoggerFactory.getLogger(HtpasswdAuthServiceFactory.class);

    private static final Pattern REALM_RX = Pattern.compile("^[a-zA-Z][a-zA-Z0-9_-]*$");

    private final ConcurrentMap<String,HtpasswdAuthService> authServicesByRealm = new ConcurrentHashMap<String,HtpasswdAuthService>();

    private void resolveFile(String filename, Consumer<Path> readFile, Consumer<String> readFileContent) throws LoginException {
        if (this.config.getHtpasswdFileContents() != null) {
            readFileContent.accept(this.config.getHtpasswdFileContents().get(filename));
        } else {
            var htpasswdDir = Optional.ofNullable(this.config.getHtpasswdDir())
            .or(() -> Optional.ofNullable(System.getProperty("login.htpasswd.etcDir"))
                .map(Path::of))
            .or(() -> Optional.ofNullable(System.getProperty("osgi.runner.etcPath"))
                .map(svcRunnerEtcPath -> Path.of(svcRunnerEtcPath).resolve("htpasswd.d")))
            .orElseThrow(() -> new LoginException("Could not find system property [login.htpasswd.etcDir] or [osgi.runner.etcPath], please check your configuration."));

            if (!Files.exists(htpasswdDir)) {
                log.warn("Htpasswd directory [{}] does not exist, check your configuration.",htpasswdDir);
            }

            readFile.accept(htpasswdDir.resolve(filename));
        }
    }

    /**
     * Get the htpasswd auth service from the directory referenced by the system property
     * <code>login.htpasswd.etcDir</code> or alternatively from
     * <code>${svc.runner.etcPath}/htpasswd.d</code>.
     *
     * @param realm The domain aka htpasswd realm.
     * @return The auth service, which should not be kept for longer than need to chsck
     *         one login operation.
     * @throws LoginException
     */
    public final IHtpasswdAuthService getServiceConfigurable(String realm) throws LoginException {
        Objects.requireNonNull(realm);

        if (!REALM_RX.matcher(realm).matches()) {
            throw new IllegalArgumentException("Inavlid realm name ["+realm+"] (must match "+REALM_RX+")");
        }

        HtpasswdAuthService svc = this.authServicesByRealm.get(realm);

        if (svc == null || svc.needsUpdate()) {
            svc = new HtpasswdAuthService();

            // Java lambdas require all captures to be final.
            final var fsvc = svc;

            resolveFile(
                realm + ".htpasswd",
                filePath -> fsvc.read(filePath.toFile()),
                content -> fsvc.readString(content)
            );

            resolveFile(
                realm + ".group",
                filePath -> fsvc.readGroupFile(filePath.toFile()),
                content -> fsvc.readGroupString(content)
            );

            resolveFile(
                realm + ".claims",
                filePath -> fsvc.readClaimsFile(filePath.toFile()),
                content -> fsvc.readClaimsString(content)
            );

            log.info("Initialized htpasswd JAAS login module for realm [{}]",realm);

            this.authServicesByRealm.put(realm,svc);
        }

        return svc;
    }

    // Backwards compatible static and unconfigurable interface.
    private static HtpasswdAuthServiceFactory DEFAULT_INSTANCE = new HtpasswdAuthServiceFactoryConfiguration()
        .build();

    public static void debugClear() {
        DEFAULT_INSTANCE = new HtpasswdAuthServiceFactoryConfiguration()
            .build();
    }

    public static IHtpasswdAuthService getService(String realm) throws LoginException {
        return DEFAULT_INSTANCE.getServiceConfigurable(realm);
    }
}
