package org.clazzes.login.htpasswd.jaas;
/*
 * Copyright 2013 Florian Zschocke
 * Copyright 2013 gitblit.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.Crypt;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.codec.digest.Md5Crypt;
import org.mindrot.jbcrypt.BCrypt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;


public class HtpasswdAuthService implements IHtpasswdAuthService {

    private static final Logger log = LoggerFactory.getLogger(HtpasswdAuthService.class);

    private final Map<String,String> htUsers;
    private final Map<String,List<HtpasswdGroup>> htGroups;
    private final Map<String,Map<String,String>> htClaims;

    private File htpasswdFile;
    private long lastModified;

    private File groupFile;
    private long groupLastModified;

    private File claimsFile;
    private long claimsLastModified;

    /**
     * Instantiate an uninitialized auth service, you need to read a htpasswd
     * file using {@link #read(File)}.
     */
    public HtpasswdAuthService() {
        this.htUsers = new HashMap<String, String>();
        this.htGroups = new HashMap<String,List<HtpasswdGroup>>();
        this.htClaims = new HashMap<String,Map<String,String>>();
    }

    /**
     * Authenticate a user based on a username and password.
     *
     * If the account is determined to be a local account, authentication
     * will be done against the locally stored password.
     * Otherwise, the configured htpasswd file is read. All current output options
     * of htpasswd are supported: clear text, crypt(), Apache MD5 and unsalted SHA-1.
     *
     * @param username
     * @param password
     * @return Whether the user/password combination is right.
     */
    @Override
    public boolean authenticate(String username, char[] password) {

        String storedPwd = this.htUsers.get(username);

        boolean authenticated = false;

        if (storedPwd != null) {
            final String passwd = new String(password);

            // test bcrypt encrypted password
            if (storedPwd.startsWith("$2y$")) {
                if (BCrypt.checkpw(passwd, storedPwd)) {
                    if (log.isDebugEnabled()) {
                        log.debug("BCrypt encoded password matched for user [{}]",username);
                    }
                    authenticated = true;
                }
            }
            else if (storedPwd.startsWith("$apr1$")) {
                if (storedPwd.equals(Md5Crypt.apr1Crypt(passwd, storedPwd))) {
                    if (log.isDebugEnabled()) {
                        log.debug("Apache MD5 encoded password matched for user [{}]",username);
                    }
                    authenticated = true;
                }
            }
            // test unsalted SHA password
            else if (storedPwd.startsWith("{SHA}")) {
                String passwd64 = Base64.encodeBase64String(DigestUtils.sha1(passwd));
                if (storedPwd.substring("{SHA}".length()).equals(passwd64)) {
                    if (log.isDebugEnabled()) {
                        log.debug("Unsalted SHA-1 encoded password matched for user [{}]",username);
                    }
                    authenticated = true;
                }
            }
            // test libc crypt() encoded password
            else if (storedPwd.equals(Crypt.crypt(passwd, storedPwd))) {
                if (log.isDebugEnabled()) {
                    log.debug("Libc crypt encoded password matched for user [{}]",username);
                }
                authenticated = true;
            }
            else {
                if (log.isDebugEnabled()) {
                    log.debug("Password for user [{}] did not match",username);
                }
            }
        }
        else {
            if (log.isDebugEnabled()) {
                log.debug("User [{}] not found in file [{}]",username,this.htpasswdFile);
            }
        }
        return authenticated;
    }

    /**
     * @param username The user to query the groups of.
     * @return The groups of the given user name read from the group file.
     */
    @Override
    public List<HtpasswdGroup> getUserGroups(String username) {

        List<HtpasswdGroup> groups = this.htGroups.get(username);

        if (groups == null) {
            return null;
        }

        return Collections.unmodifiableList(groups);
    }

    /**
     * @param username The user to query the additional claim of.
     * @return The addtional claims of the given user name read from the claims file.
     */
    @Override
    public Map<String,String> getUserClaims(String username) {

        Map<String, String> claims = this.htClaims.get(username);

        if (claims == null) {
            return null;
        }

        return Collections.unmodifiableMap(claims);
    }

    public void readString(String fileContent) {
        this.htpasswdFile = null;
        this.htUsers.clear();
        this.lastModified = -1;

        if (fileContent != null) {
            try (var reader = new StringReader(fileContent)) {
                this.htUsers.putAll(HtpasswdFileHelper.readHtpasswdFile(reader));
            } catch (IOException e) {
                log.error("Failed to read htpasswd file [{}]", fileContent, e);
            }
        }
    }

    /**
     * Reads the realm file and rebuilds the in-memory lookup tables.
     */
    public void read(File file) {

        this.htpasswdFile = file;
        this.htUsers.clear();

        if (this.htpasswdFile.exists()) {
            this.lastModified = this.htpasswdFile.lastModified();

            try (var reader = Files.newBufferedReader(file.toPath(), StandardCharsets.UTF_8)) {

                this.htUsers.putAll(HtpasswdFileHelper.readHtpasswdFile(reader));
                log.info("Successfully read [{}] user entries from htpasswd file [{}]",
                         this.htUsers.size(),
                         this.htpasswdFile);

            } catch (Exception e) {
                log.error("Failed to read htpasswd file ["+this.htpasswdFile+"]", e);
            }
        }
        else {
            log.warn("htpasswd file [{}] does not exist.",this.htpasswdFile);
            this.lastModified = -1;
        }
    }

    private void addUserToGroup(HtpasswdGroup gn, String user) {

        List<HtpasswdGroup> groups = this.htGroups.get(user);

        if (groups == null) {
            groups = new ArrayList<HtpasswdGroup>();
            this.htGroups.put(user,groups);
        }

        groups.add(gn);
    }

    private void readGroupReader(Reader file) {
        Pattern entry = Pattern.compile("^([^:]+):([^:]+)");

        try (Scanner scanner = new Scanner(file)) {

            int ngroups = 0;

            while (scanner.hasNextLine()) {
                String line = scanner.nextLine().trim();
                if (!line.isEmpty() &&  !line.startsWith("#")) {
                    Matcher m = entry.matcher(line);
                    if (m.matches()) {

                        HtpasswdGroup gn = new HtpasswdGroup(m.group(1));

                        String[] users = m.group(2).trim().split("\\s*,\\s*");

                        for (String user : users) {
                            if (!user.isEmpty()) {
                                this.addUserToGroup(gn,user);
                            }
                        }

                        ++ngroups;
                    }
                }
            }
        }
    }

    public void readGroupString(String fileContent) {
        this.groupFile = null;
        this.groupLastModified = -1;
        this.htGroups.clear();

        if (fileContent != null) {
            try {
                this.readGroupReader(new StringReader(fileContent));
                log.info("Successfully read groups from group file [{}]", fileContent);
            } catch (Exception e) {
                log.error("Failed to read group file [{}]", fileContent, e);
            }
        }
    }

    /**
     * Read a group association file composed of lines with a group name, a colon and
     * a comma-separated list of users in close relationship to the POSIX group file format.
     * @param file A group file to read.
     */
    public void readGroupFile(File file) {

        this.groupFile = file;
        this.htGroups.clear();

        if (this.groupFile != null && this.groupFile.exists()) {
            this.groupLastModified = this.groupFile.lastModified();

            try (var reader = Files.newBufferedReader(file.toPath(), StandardCharsets.UTF_8)) {
                this.readGroupReader(reader);
                log.info("Successfully read groups from group file [{}]",this.groupFile);

            } catch (Exception e) {
                log.error("Failed to read group file [{}]", this.groupFile, e);
            }
        }
        else {
            log.warn("group file [{}] does not exist.",this.groupFile);
            this.groupLastModified = -1;
        }
    }

    private void readClaimsReader(Reader reader) {
        Gson gson = new Gson();

        this.htClaims.putAll(
            gson.fromJson(reader,
                new TypeToken<Map<String,Map<String,String>>>(){
                }.getType()));
    }

    public void readClaimsString(String fileContent) {
        this.claimsFile = null;
        this.claimsLastModified = -1;
        this.htClaims.clear();

        if (fileContent != null) {
            try {
                this.readClaimsReader(new StringReader(fileContent));

                log.info("Successfully read [{}] entries from claims file [{}]",
                    this.htClaims.size(),fileContent);
            } catch (Exception e) {
                log.error("Failed to read claims file [{}]",fileContent,e);
            }
        }
    }

    /**
     * Read a claims file composed of lines with a JSON map with user names as keys and
     * a map of strings containing additional claims per user.
     * @param file A claims file to read.
     */
    public void readClaimsFile(File file) {

        this.claimsFile = file;
        this.htClaims.clear();

        if (this.claimsFile != null && this.claimsFile.exists()) {
            this.claimsLastModified = this.claimsFile.lastModified();

            try (var reader = Files.newBufferedReader(file.toPath(), StandardCharsets.UTF_8)) {
                this.readClaimsReader(reader);

                log.info("Successfully read [{}] entries from claims file [{}]",
                    this.htClaims.size(),this.claimsFile);

            }
            catch (Exception e) {
                log.error("Failed to read claims file [{}]",this.claimsFile,e);
            }
        }
        else {

            log.warn("claims file [{}] does not exist.",this.claimsFile);
            this.claimsLastModified = -1;
        }
    }

    /*
     * Method only used for unit tests. Return number of users read from htpasswd file.
     */
    public int getNumberHtpasswdUsers() {
        return this.htUsers.size();
    }

    /**
     * @return Whether this password store is outdated.
     */
    public boolean needsUpdate() {

        if (this.htpasswdFile == null) {
            return false;
        }

        if (this.htpasswdFile.exists()) {
            if (this.htpasswdFile.lastModified() > this.lastModified) {
                return true;
            }
        }
        else {
            // update needed, when the file previously existed.
            if (this.lastModified >= 0) {
                return true;
            }
        }

        // no group file loaded.
        if (this.groupFile == null) {
            return false;
        }

        if (this.groupFile.exists()) {
            return this.groupFile.lastModified() > this.groupLastModified;
        }
        else {
            // update needed, when the file previously existed.
            if (this.groupLastModified >= 0) {
                return true;
            }
        }

        // no claims file loaded.
        if (this.claimsFile == null) {
            return false;
        }

        if (this.claimsFile.exists()) {
            return this.claimsFile.lastModified() > this.claimsLastModified;
        }
        else {
            // update needed, when the file previously existed.
            return this.claimsLastModified >= 0;
        }
    }

    public File getHtpasswdFile() {
        return this.htpasswdFile;
    }

    public File getGroupFile() {
        return this.groupFile;
    }

    public File getClaimsFile() {
        return this.claimsFile;
    }

    public long getLastModified() {
        return this.lastModified;
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "[" + this.htpasswdFile + "]";
    }
}
