/**
 * $Id$
 *
 * Enterprise Service Management
 * (c) 2019 ITEG IT-Engineers GmbH https://www.iteg.at
 */
package org.clazzes.login.external;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.security.Principal;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringEscapeUtils;
import org.clazzes.util.http.ResponseHelper;
import org.clazzes.util.http.sec.HttpLoginService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A pseudo login servlet for X.509 logins.
 */
public class X509LoginServlet extends HttpServlet {

    private static final Logger log = LoggerFactory.getLogger(X509LoginServlet.class);
    
    private static final long serialVersionUID = -6824438139598839731L;

    private HttpLoginService httpLoginService;

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        
        String pi = req.getPathInfo();
        
        if (pi != null && pi.startsWith("/ctx/")) {
            
            Principal principal = this.httpLoginService.checkLogin(req);
            
            if (principal == null) {
                resp.sendError(HttpServletResponse.SC_UNAUTHORIZED);
            }
            else {
                
                boolean ok = this.httpLoginService.checkPermission(req,req.getRequestURI());
                
                if (ok) {
                    
                    try (ServletOutputStream os = resp.getOutputStream();
                            OutputStreamWriter w = new OutputStreamWriter(os,"UTF-8")) {
                           
                           w.write("<!doctype html>\n" + 
                                   "<html lang='en'>\n" + 
                                   "<head>\n" + 
                                   "<meta charset=utf-8>\n" + 
                                   "<title>X.509 Login</title>\n" + 
                                   "</head>\n" + 
                                   "<body>\n" + 
                                   "<p>Permission granted to [");
                                   
                           StringEscapeUtils.escapeHtml(w,principal.getName());

                           w.write("]</p>\n" + 
                                   "</body>\n" + 
                                   "</html>");
                       }
                }
                else {

                    log.error("Permission denied to [{}] for [{}].",req.getRequestURI(),principal.getName());
                    resp.sendError(HttpServletResponse.SC_FORBIDDEN);
                }
            }
        }
        else if ("/login".equals(pi)) {
            
            Principal principal = this.httpLoginService.checkLogin(req);
            
            if (principal == null) {
                
                log.error("Received unauthorized request to [{}].",req.getRequestURI());
                resp.sendError(HttpServletResponse.SC_UNAUTHORIZED);
            }
            else {
                
                ResponseHelper.setNoCacheHeaders(resp);
                resp.setContentType("text/html;charset=UTF-8");
                
                try (ServletOutputStream os = resp.getOutputStream();
                     OutputStreamWriter w = new OutputStreamWriter(os,"UTF-8")) {
                    
                    w.write("<!doctype html>\n" + 
                            "<html lang='en'>\n" + 
                            "<head>\n" + 
                            "<meta charset=utf-8>\n" + 
                            "<title>X.509 Login</title>\n" + 
                            "</head>\n" + 
                            "<body>\n" + 
                            "<p>Logged on as [");
                            
                    StringEscapeUtils.escapeHtml(w,principal.getName());

                    w.write("]</p>\n" + 
                            "</body>\n" + 
                            "</html>");
                }
            }
        }
        else {
            log.error("Received request to unknown sub-path [{}] of [/ext-login/org.clazzes.login.x509/].",pi);
            resp.sendError(HttpServletResponse.SC_NOT_FOUND);
        }
    }
    
    

    @Override
    public String getServletInfo() {
        
        return X509HttpLoginService.class.getSimpleName();
    }



    public void setHttpLoginService(HttpLoginService httpLoginService) {
        this.httpLoginService = httpLoginService;
    }
    
}
