/**
 * $Id$
 *
 * Enterprise Service Management
 * (c) 2019 ITEG IT-Engineers GmbH https://www.iteg.at
 */
package org.clazzes.login.external;

import java.math.BigInteger;
import java.security.Principal;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.TimeZone;

import javax.security.auth.x500.X500Principal;
import javax.servlet.http.HttpServletRequest;

import org.clazzes.util.http.RequestHelper;
import org.clazzes.util.http.sec.HttpLoginService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Validate peer certificates.
 */
public class X509HttpLoginService implements HttpLoginService {

    private static final Logger log = LoggerFactory.getLogger(X509HttpLoginService.class);
    
    public static final String CTX_URL_PREFIX = "/ext-login/org.clazzes.login.x509/ctx/";
    
    private String subjectDnHeader;
    private String issuerDnHeader;
    private String serialHeader;
    private ConfigurationService configurationService;
    private boolean isMainLoginMethod;
    
    /**
     * Parse a serial number as reported by a reverse proxy.
     * 
     * @param serial The hexadecimal serial number as reported in a
     *           <code>SSL_CLIENT_M_SERIAL</code> HTTP header with
     *           possible colons.
     * @return The parsed serial number.
     */
    public static final BigInteger parseClientSerial(String serial) {
     
        return new BigInteger(serial.replace(":",""),16);
    }

    protected X509CertRef parseCertRef(HttpServletRequest request) {
        
        String subjectDn = request.getHeader(this.subjectDnHeader);
        String issuerDn = request.getHeader(this.issuerDnHeader);
        String serialString = request.getHeader(this.serialHeader);
        
        if (subjectDn == null) {
            if (this.isMainLoginMethod) {
                log.error("Missing Subject DN in request to [{}]", RequestHelper.describeRequest(request));
            }
            return null;
        }
        if (!this.isMainLoginMethod) {
            log.info("Found Subject DN in request to [{}]",RequestHelper.describeRequest(request));
        }

        // if subject DN is set, the issuer DN should always be set as well
        if (issuerDn == null) {
            log.error("Missing Issuer DN in request to [{}]",RequestHelper.describeRequest(request));
            return null;
        }

        // if subject DN is set, the serial should always be set as well
        if (serialString == null) {
            log.error("Missing Serial in request to [{}]",RequestHelper.describeRequest(request));
            return null;
        }

        BigInteger serial;
        
        try {
            serial = parseClientSerial(serialString);
        }
        catch (NumberFormatException e) {
            log.error("Malformatted serial number [{}] in request to [{}]",
                    serialString,RequestHelper.describeRequest(request));
            return null;
        }
        
        return new X509CertRef(subjectDn,issuerDn,serial);
    }
    
    @Override
    public String getLoginUrl() {
        
        return "/ext-login/org.clazzes.login.x509/login";
    }

    @Override
    public Principal checkLogin(HttpServletRequest request) {
        
        X509CertRef ref = this.parseCertRef(request);
                
        if (ref == null) {
            return null;
        }
        
        IssuerConfig issuerConfig = this.configurationService.getIssuerConfig(ref.getIssuerDn());
        
        if (issuerConfig == null) {
            if (this.isMainLoginMethod) {
                log.error("Request with certificate [{}] to [{}] refers to an unknown issuer DN.",
                        ref, RequestHelper.describeRequest(request));
            } else {
                log.info("Request with certificate [{}] to [{}] refers to an unknown issuer DN.",
                        ref, RequestHelper.describeRequest(request));
            }
            return null;
        }
        
        if (!issuerConfig.checkCertStatus(ref)) {
            return null;
        }
        
        log.info("Got request with certificate [{}] to [{}]",
                ref,RequestHelper.describeRequest(request));
        try {
            return new X500Principal(ref.getSubjectDn());
        }
        catch(IllegalArgumentException e) {
            
            log.error("Request with certificate [{}] to [{}] contains a malformatted subject DN.",
                    ref,RequestHelper.describeRequest(request));
            
            if (log.isDebugEnabled()) {
                log.debug("Detailed IAE for malformatted subject DN",e);
            }
            
            return  null;
        }
    }

    @Override
    public List<? extends Principal> checkLoginGroups(HttpServletRequest request) {
        
        return null;
    }

    @Override
    public Locale getLocale(HttpServletRequest request) {
       
        return request.getLocale();
    }

    @Override
    public TimeZone getTimeZone(HttpServletRequest request) {
        
        return TimeZone.getDefault();
    }

    @Override
    public boolean checkPermission(HttpServletRequest request, String context) {
        
        if (context.equals(this.getLoginUrl())) {
            
            return true;
        }
        
        if (context.startsWith(CTX_URL_PREFIX)) {
            
            String ctx = context.substring(CTX_URL_PREFIX.length());
            
            Set<String> wl = this.configurationService.getWhiteList(ctx);
            
            if (wl != null) {
            
                String subjectDn = request.getHeader(this.subjectDnHeader);
                
                return wl.contains(subjectDn);
            }
        }
        
        return false;
    }

    @Override
    public void logout(HttpServletRequest request) {
        log.info("X.509 logout ist not implemented.");
    }

    public void setSubjectDnHeader(String subjectDnHeader) {
        this.subjectDnHeader = subjectDnHeader;
    }

    public void setIssuerDnHeader(String issuerDnHeader) {
        this.issuerDnHeader = issuerDnHeader;
    }

    public void setSerialHeader(String serialHeader) {
        this.serialHeader = serialHeader;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    public void setIsMainLoginMethod(Boolean isMainLoginMethod) {
        this.isMainLoginMethod = isMainLoginMethod == null || isMainLoginMethod;
    }
}
