/**
 * $Id$
 *
 * Enterprise Service Management
 * (c) 2019 ITEG IT-Engineers GmbH https://www.iteg.at
 */
package org.clazzes.login.external;

import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.security.KeyStore;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509CRLEntry;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.Set;

import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

import org.clazzes.util.sched.ITimedJob;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A configuration for an issuer.
 */
public class IssuerConfig implements Runnable, ITimedJob {

    private static final Logger log = LoggerFactory.getLogger(IssuerConfig.class);
    
    private final String crlDistributionPoint;
    private final String crlVerificationCertificate;
    private final Set<String> subjectDnBlacklist;
    private final long refreshCrlMinutes;
    private final long refreshCrlMinutesOnFailure;
    
    // thread-safe state
    private Long nextCrlLoad;
    private X509CRL crl;
    
    public IssuerConfig(String crlDistributionPoint,
            String crlVerificationCertificate,
            Set<String> subjectDnBlacklist,
            long refreshCrlMinutes, long refreshCrlMinutesOnFailure) {
        super();
        this.crlDistributionPoint = crlDistributionPoint;
        this.crlVerificationCertificate = crlVerificationCertificate;
        this.subjectDnBlacklist = subjectDnBlacklist;
        this.refreshCrlMinutes = refreshCrlMinutes;
        this.refreshCrlMinutesOnFailure = refreshCrlMinutesOnFailure;
        this.nextCrlLoad = crlDistributionPoint == null ? null : System.currentTimeMillis() + 1000L;
    }

    public synchronized X509CRL getCrl() {
        return this.crl;
    }

    public synchronized void setCrl(X509CRL crl) {
      
        if (crl == null) {
            this.nextCrlLoad = System.currentTimeMillis() + this.refreshCrlMinutesOnFailure * 60000L;
        }
        else {
            
            long now = System.currentTimeMillis();
            long updateMillis = System.currentTimeMillis() + this.refreshCrlMinutes * 60000L;
            
            Date d = crl.getNextUpdate();
             
            if (d != null && d.getTime() < updateMillis) {
                
                if (d.getTime() > now) {
                    updateMillis = d.getTime();
                }
                else {
                    log.warn("Next update [{}] of CRL for [{}] is in the past, using failure refresh interval.",d,crl.getIssuerX500Principal());
                    updateMillis = System.currentTimeMillis() + this.refreshCrlMinutesOnFailure * 60000L;
                }
            }
            
            log.info("Next CRL refresh for [{}] is [{}].",crl.getIssuerX500Principal(),new Date(updateMillis));
            this.nextCrlLoad = updateMillis;
        }
        
        this.crl = crl;        
    }

    public String getCrlDistributionPoint() {
        return this.crlDistributionPoint;
    }

    public Set<String> getSubjectDnBlacklist() {
        return this.subjectDnBlacklist;
    }

    public boolean checkCertStatus(X509CertRef ref) {
        
        if (this.subjectDnBlacklist != null &&
                this.subjectDnBlacklist.contains(ref.getSubjectDn())) {
            
            log.error("Certificate [{}] is blacklisted.",ref);
            return false;
        }
        
        X509CRLEntry entry = null;
        synchronized (this) {
            if (this.crl == null) {
                if (this.crlDistributionPoint != null) {
                    log.error("Certificate [{}] is rejected, because the issuer CRL is not yet loaded.",ref);
                    return false;
                }
                else {
                    // no CRL configured.
                    return true;
                }
            }
            
            entry = this.crl.getRevokedCertificate(ref.getSerial());
        }
        
        if (entry == null) {
            return true;
        }
        else {
            log.error("Certificate [{}] has been revoked at [{}] with reason [{}].",
                    new Object[]{ref,entry.getRevocationDate(),entry.getRevocationReason()});
                return false;
        }

    }

    @Override
    public synchronized Long getNextExecutionDelay() {
        
        return this.nextCrlLoad == null ? null : (this.nextCrlLoad.longValue()-System.currentTimeMillis());
    }

    @Override
    public void run() {
        
        X509CRL verifiedCrl = null;
        
        try {
            
            
            log.info("Loading CRL from [{}]",this.crlDistributionPoint);
            URL url = new URL(this.crlDistributionPoint);
            
            URLConnection conn = url.openConnection();
            
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            
            X509CRL crl;
            
            try (InputStream is = conn.getInputStream()) {
                
                crl = (X509CRL) cf.generateCRL(is);
            }
            
            log.info("Successfully loaded CRL from [{}]",this.crlDistributionPoint);
            
            if (this.crlVerificationCertificate == null) {
            
                TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
                trustManagerFactory.init((KeyStore) null);
                
                for (TrustManager trustManager : trustManagerFactory.getTrustManagers()) {
                    
                    if (trustManager instanceof X509TrustManager) {
                        
                        X509TrustManager x509TrustManager = (X509TrustManager)trustManager;
                        
                        for (X509Certificate cert : x509TrustManager.getAcceptedIssuers()) {
                            
                            if (cert.getSubjectX500Principal().equals(crl.getIssuerX500Principal())) {
                                
                                log.info("Verifying CRL from [{}] with Certificate [{}].",
                                        this.crlDistributionPoint,cert.getSubjectX500Principal());
                                
                                cert.checkValidity();
                                
                                crl.verify(cert.getPublicKey());
                                
                                log.info("Successfully verified CRL from [{}]",this.crlDistributionPoint);
                                verifiedCrl = crl;
                                break;
                            }
                        }
                    }
                }
            }
            else {
                
                X509Certificate cert;
                
                try (InputStream is = new FileInputStream(this.crlVerificationCertificate)) {
                    
                    cert = (X509Certificate)cf.generateCertificate(is);
                }
                

                log.info("Verifying CRL from [{}] with Certificate [{}] from [{}].",
                        new Object[] {this.crlDistributionPoint,cert.getSubjectX500Principal(),this.crlVerificationCertificate});
                
                cert.checkValidity();
                
                crl.verify(cert.getPublicKey());
                
                log.info("Successfully verified CRL from [{}]",this.crlDistributionPoint);
                verifiedCrl = crl;
            }
            
            if (verifiedCrl == null) {
                log.error("Cannot find a signing certificate for CRL from [{}].",this.crlDistributionPoint);
            }
            
        } catch (Exception e) {
            log.error("Error loading CRL from URL ["+this.crlDistributionPoint+"]",e);
        }
        
        this.setCrl(verifiedCrl);
    }
}
